require 'rails_helper'

describe V1::PeopleController do
  describe '#show' do
    before :all do
      @org = FactoryGirl.create(:organization, :with_teams, :with_people)
      @person = FactoryGirl.create(:person, :with_teams)
      @person.organizations << @org
    end

    it 'looks up a Person by their user_id instead of Person.id' do
      get "/v1/people/#{@person.user_id}"
      expect(response.status).to eq(200)
      expect(json['user_id']).to eq(@person.user_id)
    end

    for embed in ['teams', 'organizations'] do
      it "should embed #{embed}" do
        get "/v1/people/#{@person.user_id}"
        expect(response.status).to eq(200), 'GET failed: '+JSON.pretty_generate(json)
        expect(json.keys).to include(embed)
        expect(json[embed].collect{|t| t['id']}).to include(@person.send(embed)[0].id)
      end
    end
  end

  describe '#me' do
    it 'replies with the Person making the request' do
      person = FactoryGirl.create(:person)
      get '/v1/people/me',
        params: {user_id: FactoryGirl.create(:person).user_id},
        headers: {'User-ID' => person.user_id}
      expect(response.status).to eq(200)
      expect(json['user_id']).to eq(person.user_id)
    end
  end

  it_behaves_like 'an image uploader for', :person, :photo, {
    id_attr: :user_id,
    headers: admin_headers,
  }

  describe '#update' do
    it 'only allows admin and self' do
      person = FactoryGirl.create(:person)

      # No user (Not Authenticated instead of Forbidden)
      patch "/v1/people/#{person.user_id}"
      expect(response.status).to eq(401)

      # Wrong user
      patch "/v1/people/#{person.user_id}",
        headers: {'User-ID' => SecureRandom.uuid}
      expect(response.status).to eq(403)

      # Self
      patch "/v1/people/#{person.user_id}",
        headers: {'User-ID' => person.user_id}
      expect(response.status).to eq(200)

      # Admin
      patch "/v1/people/#{person.user_id}",
        headers: admin_headers
      expect(response.status).to eq(200)
    end
  end
end
