require 'rails_helper'

describe V1::SeriesController do
  it_behaves_like 'a CRUD API', { for: :series }, -> {{
    create: FactoryGirl.build(:series).attributes.merge({
      matches: [{
        opponents: FactoryGirl.create_list(:team, 2).collect{|t|{
          contender_id: t.id,
          contender_type: 'Team'
        }}
      }],
      skip_curse: true
    }),
    update: { type: 'Series::Bo5' }
  }}

  describe '#index' do
    it 'finds all series a user is involved in when given a user_id' do
      # Helper to make creating lots of test Series less annoying:
      def new_series(game, assoc_series, assoc_match, *contenders)
        series_params = {game: game}
        series_params[:series_contenders] = contenders if assoc_series
        series_params[:match_contenders] = contenders if assoc_match
        FactoryGirl.create(:series, series_params).id
      end

      game = FactoryGirl.create(:game)
      me = FactoryGirl.create(:person)
      my_teams = FactoryGirl.create_list(:team_with_people, 2).each {|t| t.add_member!(me)}
      other_teams = FactoryGirl.create_list(:team_with_people, 2)

      # Values for `series` are `[expected, description]`.
      series = {}
      [true,false].repeated_permutation(2).each do |assoc_series, assoc_match|
        tf_tag = " (series=#{assoc_series}, match=#{assoc_match})"
        expected = assoc_series || assoc_match

        # Expected results:
        series[new_series(game, assoc_series, assoc_match, me)] =
            [expected, "in which I'm a solo participant"+tf_tag]
        series[new_series(game, assoc_series, assoc_match, my_teams[0], other_teams[0])] =
            [expected, "in which I'm on a team (1)"+tf_tag]
        series[new_series(game, assoc_series, assoc_match, my_teams[1], other_teams[1])] =
            [expected, "in which I'm on a team (2)"+tf_tag]
        series[new_series(game, assoc_series, assoc_match, my_teams[0], my_teams[1])] =
            [expected, "in which I'm on both teams"+tf_tag]

        # Potential false positives:
        series[new_series(game, assoc_series, assoc_match)] =
            [false, "between two randos"+tf_tag]
        series[new_series(game, assoc_series, assoc_match, other_teams[0], other_teams[1])] =
            [false, "between two teams I'm not on"+tf_tag]
        series[new_series(FactoryGirl.create(:game), assoc_series, assoc_match, me)] =
            [false, "for a different game"+tf_tag]
      end

      get '/v1/series', params: {user_id: me.user_id, game_id: game.id}
      expect(response.status).to eq(200)

      reply_ids = json_ids
      series.each do |series_id, meta|
        expected, description = meta
        if reply_ids.include?(series_id) != expected
          fail "Series #{description} should#{expected ? "" : " not"} be in reply"
        end
      end
    end

    describe 'tournament association' do
      before :all do
        @series_ids = {}
        @tournaments = FactoryGirl.create_list(:tournament, 5).each do |tournament|
          group_stage = tournament.stages.detect {|stage| stage.name == 'Group Stage'}
          @series_ids[tournament.id] = FactoryGirl.create_list(:series, 3, tournament_stage: group_stage).map(&:id).sort
        end
      end

      it 'can filter by tournament_id' do
        tournament_id = @tournaments[2].id
        get '/v1/series', params: {tournament_id: tournament_id}
        expect(response.status).to eq(200), ->{pp json}
        expect(json_ids.sort).to eq(@series_ids[tournament_id])
      end

      it 'can filter by tournament_ids' do
        tournament_ids = @tournaments.values_at(0,2,4).map(&:id)
        series_ids = @series_ids.values_at(*tournament_ids).flatten.sort

        # As array:
        get '/v1/series', params: {tournament_ids: tournament_ids}
        expect(response.status).to eq(200), ->{pp json}
        expect(json_ids.sort).to eq(series_ids)

        # As string:
        get '/v1/series', params: {tournament_ids: tournament_ids.map(&:to_s).join(',')}
        expect(response.status).to eq(200), ->{pp json}
        expect(json_ids.sort).to eq(series_ids)
      end
    end

    describe 'scheduled_at' do
      before :all do
        times = (-3..3).map(&:day).map(&:from_now)
        @series = times.map {|time| FactoryGirl.create(:series, scheduled_at: time)}
        @series_ids = @series.map(&:id)
        @today = Time.now.to_s  # intentionally not at start of day
      end

      it 'orders series chronologically' do
        get '/v1/series', params: {ids: @series_ids}
        expect(response.status).to eq(200)
        expect(json_ids.sort).to eq(@series_ids.sort)
        reply_times = json.map {|series| series['starts_at']}
        expect(reply_times).to eq(reply_times.sort)
      end

      it 'filters based on "on"' do
        series_id = @series_ids[3]
        get '/v1/series', params: {ids: @series_ids, on: @today}
        expect(response.status).to eq(200)
        expect(json_ids).to eq([series_id])
      end

      it 'filters based on "from"' do
        get '/v1/series', params: {ids: @series_ids, from: @today}
        expect(response.status).to eq(200)
        expect(json_ids).to eq(@series_ids[3..6])
      end

      it 'filters based on "to"' do
        get '/v1/series', params: {ids: @series_ids, to: @today}
        expect(response.status).to eq(200)
        expect(json_ids).to eq(@series_ids[0..3])
      end

      it 'filters based on both "from" and "to"' do
        get '/v1/series', params: {ids: @series_ids, from: 1.day.ago.to_s, to: 1.day.from_now.to_s}
        expect(response.status).to eq(200)
        expect(json_ids).to eq(@series_ids[2..4])
      end
    end
  end

  describe '#show' do
    it 'renders a Bo5 series using SeriesPresenter' do
      series = FactoryGirl.create(:bo5_series)
      expect(V1::SeriesPresenter).to receive(:new).and_call_original
      get "/v1/series/#{series.id}"
      expect(response.status).to eq(200)
      expect(json['type']).to eq('Series::Bo5')
      expect(json).to include('winner_id')
    end
  end

  describe '#create' do
    before :all do
      @curse_server_orig, @curse_auth_orig = ENV.values_at('CURSE_SERVER', 'CURSE_AUTH_TOKEN')
      ENV['CURSE_SERVER'] = @server_id = SecureRandom.uuid
      ENV['CURSE_AUTH_TOKEN'] = 'a totally secure token'
    end

    after :all do
      ENV['CURSE_SERVER'] = @curse_server_orig
      ENV['CURSE_AUTH_TOKEN'] = @curse_auth_orig
    end

    it 'should include an invite link to a Curse chat room' do
      tournament = FactoryGirl.create(:tournament)
      teams = FactoryGirl.create_list(:team, 2, game: tournament.game)
      opponents = teams.map {|team| {contender_id: team.id, contender_type: 'Team'}}
      params = {
        tournament_stage_id: tournament.stages[0].id,
        opponents: opponents,
        matches: [{opponents: opponents}],
      }

      room_name = "#{tournament.game.abbr}: " + teams.map(&:name).sort.join(' vs ')
      channel = FactoryGirl.build(:curse_channel, group_title: room_name)
      stub_request(:get, "https://groups-v1.curseapp.net/groups/#{@server_id}").
        to_return(body: {'Channels' => [channel]}.to_json)
      invite = Faker::Internet.url('example.com')
      stub_request(:post, "https://groups-v1.curseapp.net/servers/#{@server_id}/invites").
        to_return(body: {'InviteUrl' => invite}.to_json)

      post '/v1/series', headers: admin_headers, params: params
      expect(response.status).to eq(200), response.body
      expect(json.dig('settings','twitch_voice_url')).to eq(invite)
    end
  end

  describe '#update' do
    before :each do
      @series = FactoryGirl.create(:bo5_series)
    end

    it 'accepts nested attributes for matches through matches' do
      teams = FactoryGirl.create_list(:team, 2)
      patch "/v1/series/#{@series.id}", params: {
        matches: @series.matches.collect{|t| {id: t.id, _destroy: true}} +
          [{
            opponents: teams.collect{|t|{
              contender_id: t.id,
              contender_type: 'Team'
            }}
          }]
      }, headers: admin_headers, as: :json
      expect(response).to be_ok, "not ok: #{json}"
      @series.reload
      expect(@series.team_ids.sort).to eq(teams.collect(&:id).sort)
    end

    it 'validates opponents through matches' do
      match = @series.matches.first
      opp = match.opponents.first
      patch "/v1/series/#{@series.id}", params: {
        matches: [{id: match.id, opponents: {id: opp.id, _destroy: true}}]
      }, headers: admin_headers, as: :json
      expect(response.status).to eq(400)
      expect(json['errors'].keys).to include('matches')
    end
  end
end
