require 'rails_helper'

describe V1::TeamInvitationsController do
  describe 'POST /team_invitations' do
    before :each do
      @org = FactoryGirl.create(:organization, email_domains: ['example.com'])
      @team = FactoryGirl.create(:team_with_people, organization: @org)
      @owner = @team.members('owner').first.person
    end

    it 'should require the team to be affiliated with an organization' do
      @team.update!(organization: nil)
      post '/v1/team_invitations',
        params: { team_id: @team.id, email: 'foo@example.com' },
        headers: { 'User-ID': @owner.user_id }
      expect(response.status).to eq(400), ->{pp json}
      expect(json['error']).to match(/affiliated.*organization/)
    end

    it "should require the email to have a domain associated with the team's organization" do
      post '/v1/team_invitations',
        params: { team_id: @team.id, email: 'foo@nope.example.com' },
        headers: { 'User-ID': @owner.user_id }
      expect(response.status).to eq(400), ->{pp json}
      expect(json['error']).to match(/member.*organization/)
    end

    it 'should require a valid team' do
      post '/v1/team_invitations',
        params: { team_id: 0, email: 'foo@example.com' },
        headers: { 'User-ID': @owner.user_id }
      expect(response.status).to eq(404)

      post '/v1/team_invitations',
        params: { team_id: @team.id, email: 'foo@example.com' },
        headers: { 'User-ID': @owner.user_id }
      expect(response.status).to eq(200), ->{pp json}
    end

    it 'should return the created invitation' do
      post '/v1/team_invitations',
        params: { team_id: @team.id, email: 'foo@example.com' },
        headers: { 'User-ID': @owner.user_id }
      expect(response.status).to eq(200), ->{pp json}
      expect(json['team']['id']).to eq(@team.id)
    end
  end

  describe 'PATCH /team_invitations/:token/accept' do
    before :all do
      @invitation = FactoryGirl.create(:team_invitation)
      @invitation.team.update_attribute(:join_token, 'jointoken')
      @person = FactoryGirl.create(:person, user_id: SecureRandom.uuid)
    end

    it 'should require_user!' do
      patch "/v1/team_invitations/#{@invitation.token}/accept"
      expect(response.status).to eq(401)
    end

    it 'should accept the invitation on behalf of the current user' do
      patch "/v1/team_invitations/#{@invitation.token}/accept",
        params: {}, headers: { 'User-ID' => @person.user_id }
      expect(response).to be_ok
      expect(@invitation.team.people).to include(@person)
    end

    it 'should add the current user to a team matching the teams join_token' do
      patch "/v1/team_invitations/#{@invitation.team.join_token}/accept",
        params: {}, headers: { 'User-ID' => @person.user_id }
      expect(response).to be_ok
      expect(@invitation.team.people).to include(@person)
    end

    it 'should raise error if neither team nor invitation are found' do
      patch "/v1/team_invitations/lel/accept",
        params: {}, headers: { 'User-ID' => @person.user_id }
      expect(response).to_not be_ok
      expect(response.status).to eq(404)
    end

    it 'should not find invitation if it has been used' do
      @invitation.accept!(@person)
      expect(@invitation.used).to be(true)
      patch "/v1/team_invitations/#{@invitation.token}/accept",
        params: {}, headers: { 'User-ID' => @person.user_id }
      expect(response).to_not be_ok
      expect(response.status).to eq(404)
    end
  end

  describe 'DELETE /team_invitations/:id' do
    before :each do
      @invitation = FactoryGirl.create(:team_invitation)
      @owner = @invitation.team.add_member!(FactoryGirl.create(:person), ['owner']).person
    end

    it 'should destroy the invitation' do
      delete "/v1/team_invitations/#{@invitation.id}",
          headers: { 'User-ID' => @owner.user_id }
      expect(response.status).to eq(200), "DELETE failed: "+JSON.pretty_generate(json)
      expect{@invitation.reload}.to raise_error(ActiveRecord::RecordNotFound)
    end

    it 'should not destroy used invitations' do
      person = FactoryGirl.create(:person)
      @invitation.accept!(person)
      delete "/v1/team_invitations/#{@invitation.id}",
        headers: { 'User-ID': @owner.user_id }
      expect(response.status).to eq(404)
      expect{@invitation.reload}.to_not raise_error
    end

    it 'return the deleted invitation' do
      delete "/v1/team_invitations/#{@invitation.id}",
        headers: { 'User-ID': @owner.user_id }
      expect(response).to be_ok
      expect(json['id']).to eq(@invitation.id)
    end

    it 'should not be associated with the returned team' do
      delete "/v1/team_invitations/#{@invitation.id}",
        headers: { 'User-ID': @owner.user_id }
      expect(response).to be_ok
      expect(json['team_id']).to be_nil
      expect(json['team']['team_invitation_ids']).to_not include(@invitation.id)
    end
  end
end
