require 'rails_helper'

describe V1::TeamMembersController do
  describe 'DELETE /teams/:team_id/members/:member_id' do
    before :all do
      @team = FactoryGirl.create(:team_with_people)
      @owner = @team.members('owner').first.person
    end

    it 'should respond with a 404 if there is no member with the given member_id' do
      bad_id = TeamMember.select(:id).order('id DESC').first.id + 10
      delete "/v1/teams/#{@team.id}/members/#{bad_id}",
        headers: { 'User-ID': @owner.user_id }
      expect(response.status).to eq(404)
    end

    it 'should respond with a 404 if the team was not found' do
      delete "/v1/teams/#{SecureRandom.uuid}/members/#{@owner.id}",
        headers: { 'User-ID': @owner.user_id }
      expect(response.status).to eq(404)
    end

    it 'should remove the person matching the given member_id from the team' do
      member = @team.members.first
      person = member.person
      delete "/v1/teams/#{@team.id}/members/#{member.id}",
        headers: { 'User-ID': @owner.user_id }
      expect(response.status).to eq(200)
      @team.reload
      expect(@team.members).to_not include(member)
      expect(@team.people).to_not include(person)
      expect(person).to be_persisted
    end

    it 'should let non-privileged members remove themselves' do
      member = @team.members.detect {|m| m.roles.empty?}
      delete "/v1/teams/#{@team.id}/members/#{member.id}",
        headers: { 'User-ID': member.person.user_id }
      expect(response.status).to eq(200)
      @team.reload
      expect(@team.members).to_not include(member)
      expect(@team.people).to_not include(member.person)
    end
  end

  describe 'PATCH /teams/:team_id/members/:member_id' do
    before :each do
      @team = FactoryGirl.create(:team_with_people)
      @owner = @team.members('owner').first.person
    end

    it 'can clear team member roles' do
      member = @team.members[1]
      member.update!(roles: ['foo','bar','baz'])
      patch "/v1/teams/#{@team.id}/members/#{member.id}",
        headers: { 'User-ID': @owner.user_id },
        params: { roles: [] }, as: :json
      expect(response.status).to eq(200)
      expect(json['roles'].length).to eq(0)
      expect(member.reload.roles.length).to eq(0)
    end
  end
end
