require 'rails_helper'

describe V1::TeamUsersController do
  before :each do
    @team = FactoryGirl.create(:team_with_people)
  end

  context '#index' do
    it 'should reply with all the users on the team' do
      get "/v1/teams/#{@team.id}/users"
      expect(response.status).to eq(200)
      expect(json).to be_a(Array)
      expect(json.map{|u|u['user_id']}.sort).to eq(@team.people.map(&:user_id).sort)
    end

    it 'should reply with 404 if the team does not exist' do
      get "/v1/teams/#{SecureRandom.uuid}"
      expect(response.status).to eq(404)
    end
  end

  context '#create' do
    it 'should add a user to the team' do
      new_person = FactoryGirl.create(:person)
      post "/v1/teams/#{@team.id}/users",
        params: {user_id: new_person.user_id},
        headers: admin_headers
      expect(response.status).to eq(200)
      expect(json['user_id']).to eq(new_person.user_id)
      @team.reload
      expect(@team.people).to include(new_person)
      expect(@team.members.map(&:person)).to include(new_person)
      expect(TeamMember.exists?(team_id: @team.id, person_id: new_person.id)).to eq(true)
    end

    it 'should reply with 404 if the team does not exist' do
      post "/v1/teams/#{SecureRandom.uuid}/users",
        params: {user_id: SecureRandom.uuid},
        headers: admin_headers
      expect(response.status).to eq(404)
    end

    it 'should require admin privileges' do
      post "/v1/teams/#{@team.id}/users",
        params: {user_id: SecureRandom.uuid},
        headers: {'User-ID' => SecureRandom.uuid}
      expect(response.status).to eq(403)

      post "/v1/teams/#{@team.id}/users",
        params: {user_id: SecureRandom.uuid},
        headers: {'User-ID' => @team.members('owner').first.person.user_id}
      expect(response.status).to eq(403)
    end
  end

  context '#destroy' do
    it 'should reply with a 404 for a team that does not exist' do
      delete "/v1/teams/#{SecureRandom.uuid}/users/#{@team.people.first.user_id}",
        headers: admin_headers
      expect(response.status).to eq(404)
    end

    it 'should reply with a 404 for a user who is not on the team' do
      delete "/v1/teams/#{@team.id}/users/#{SecureRandom.uuid}",
        headers: admin_headers
      expect(response.status).to eq(404)
    end

    it 'should remove a user from the team' do
      person = @team.people.first
      delete "/v1/teams/#{@team.id}/users/#{person.user_id}",
        headers: admin_headers
      expect(response.status).to eq(200)
      expect(json['user_id']).to eq(person.user_id)
      @team.reload
      expect(@team.people.length).to be > 0
      expect(@team.members.length).to be > 0
      expect(@team.people).to_not include(person)
      expect(@team.members.map(&:person)).to_not include(person)
      expect(TeamMember.exists?(team_id: @team.id, person_id: person.id)).to eq(false)
      expect(person).to be_persisted
      expect(@team).to be_persisted
    end

    it 'should let a user remove himself from the team' do
      person = @team.people.first
      delete "/v1/teams/#{@team.id}/users/#{person.user_id}",
        headers: {'User-ID' => person.user_id}
      expect(response.status).to eq(200)
      expect(json['user_id']).to eq(person.user_id)
      @team.reload
      expect(@team.people.length).to be > 0
      expect(@team.members.length).to be > 0
      expect(@team.people).to_not include(person)
      expect(@team.members.map(&:person)).to_not include(person)
      expect(TeamMember.exists?(team_id: @team.id, person_id: person.id)).to eq(false)
      expect(person).to be_persisted
      expect(@team).to be_persisted
    end

    it 'should not allow non-owners/captains to remove members' do
      delete "/v1/teams/#{@team.id}/users/#{@team.people.first.user_id}",
        headers: {'User-ID' => SecureRandom.uuid}
      expect(response.status).to eq(403)
    end

    it 'should allow owners to remove members' do
      delete "/v1/teams/#{@team.id}/users/#{@team.people.first.user_id}",
        headers: {'User-ID' => @team.members('owner').first.person.user_id}
      expect(response.status).to eq(200)
    end

    it 'should allow captains to remove members' do
      delete "/v1/teams/#{@team.id}/users/#{@team.people.first.user_id}",
        headers: {'User-ID' => @team.members('captain').first.person.user_id}
      expect(response.status).to eq(200)
    end
  end
end
