require 'rails_helper'

describe V1::TeamsController do
  it_behaves_like 'a #index API', { for: :team }
  it_behaves_like 'a #show API', { for: :team }
  it_behaves_like 'a #update API', { for: :team }, ->{{
    update: {name: 'new team name'}
  }}

  describe 'GET /teams' do
    before :all do
      @person = FactoryGirl.create(:person_with_teams)
    end

    it 'returns teams a person belongs to when given a user_id' do
      # Make sure there's a team that the user isn't a member of
      FactoryGirl.create(:team)
      get '/v1/teams',
        params: {user_id: @person.user_id},
        headers: {}
      for team in json do
        expect(team['team_members'].collect{|p| p['user_id']}).to include(@person.user_id)
      end
    end

    it 'should not include hidden teams by default' do
      teams = FactoryGirl.create_list(:team, 5)
      hidden_teams = FactoryGirl.create_list(:team, 5, :hidden)
      all_teams = teams + hidden_teams

      get "/v1/teams", params: {ids: all_teams.map(&:id)}
      expect(response.status).to eq(200), ->{pp json}
      expect(json_ids.sort).to eq(teams.map(&:id).sort)

      get "/v1/teams", params: {ids: all_teams.map(&:id), hidden: true}
      expect(response.status).to eq(200), ->{pp json}
      expect(json_ids.sort).to eq(all_teams.map(&:id).sort)
    end
  end

  describe 'GET /teams/:id' do
    before :all do
      @team = FactoryGirl.create(:team)
    end

    it 'should be able to look up a team by slug' do
      expect(@team.slug).to_not be_blank
      get "/v1/teams/#{@team.slug}"
      expect(response).to be_ok
      expect(json['id']).to eq(@team.id)
      expect(json['slug']).to eq(@team.slug)
    end

    it 'should find hidden teams by default' do
      team = FactoryGirl.create(:team, :hidden)
      get "/v1/teams/#{team.id}"
      expect(response.status).to eq(200), ->{pp json}
      expect(json['id']).to eq(team.id)
      expect(json['hidden']).to eq(true)
    end

    it 'should return empty logo urls if the team has no logo set' do
      expect(@team.logo.url).to be_nil
      get "/v1/teams/#{@team.id}"
      expect(response).to be_ok
      [:original, :preview, :thumb].each do |type|
        expect(json['logo'][type.to_s]).to be_empty
      end
    end

    it 'should prefix logo urls w/ request.base_url when carrierwave returns a relative URI' do
      @team.update_attribute(:logo, File.open('spec/support/files/glitch.png'))
      expect(@team.logo.url).to_not match(/^http/)
      get "/v1/teams/#{@team.id}"
      expect(response).to be_ok
      [:original, :preview, :thumb].each do |type|
        expect(json['logo'][type.to_s]).to match(/^http:\/\/www\.example\.com/)
      end
    end
  end

  describe 'POST /teams' do
    before :all do
      @user_id = SecureRandom.uuid
      @verified_user = FactoryGirl.create(:organization_with_people, people_count: 1).people.first
    end

    it 'requires a user' do
      post '/v1/teams',
        params: {name: 'testing'},
        headers: {}
      expect(response.status).to eq(401)
    end

    it 'requires that the user have a verified email' do
      post '/v1/teams',
        params: {name: 'testing'},
        headers: {'user-id' => @user_id}
      expect(response.status).to eq(403)
      expect(json['error']).to match(/verified email/)
    end

    # TODO: Choose a sensible default when people can _actually_ be part of many orgs.
    it "defaults to the user's organization if not specified" do
      @verified_user.update!(organizations: FactoryGirl.create_list(:organization, 1))
      post '/v1/teams',
        params: {name: 'testing'},
        headers: {'user-id' => @verified_user.user_id}
      expect(response.status).to eq(200), ->{pp json}
      expect(json['organization_id']).to eq(@verified_user.organization_ids.first)
    end

    it "forbids creating a team for an organization the owner is not a member of" do
      post '/v1/teams',
        params: {name: 'testing', 'organization_id': SecureRandom.uuid},
        headers: {'user-id' => @verified_user.user_id}
      expect(response.status).to eq(403)
      expect(json['error']).to match(/organization.*not.*member/)
    end

    it 'requires a name' do
      post '/v1/teams',
        params: {},
        headers: {'user-id' => @verified_user.user_id}
      expect(json['errors']['name'].join).to match(/can't be blank/)
    end

    it 'creates a valid Team with current user as member' do
      post '/v1/teams',
        params: {name: 'I R TEAM'},
        headers: {'user-id' => @verified_user.user_id}
      expect(response.status).to eq(200), ->{pp json}
      team = Team.find(json['id'])
      expect(team).to be_valid
      expect(team.people).to include(@verified_user)
    end

    it 'can create hidden teams using `settings["hidden"]`' do
      post '/v1/teams',
        params: {name: 'The Cheating Cheaters', settings: {hidden: true}},
        headers: admin_headers
      expect(response.status).to eq(200), ->{pp json}
      team = Team.unscoped.find(json['id'])
      expect(team.name).to eq('The Cheating Cheaters')
      expect(team.hidden).to eq(true)
      expect(team.settings['hidden']).to eq(true)
    end
  end

  describe 'PATCH /teams/:id' do
    before :all do
      @team = FactoryGirl.create(:team_with_people)
      @owner = @team.members('owner').first.person
    end

    it 'allows updates to name' do
      patch "/v1/teams/#{@team.id}",
        params: { name: 'something else' },
        headers: { 'User-ID': @owner.user_id }
      expect(response.status).to eq(200)
      @team.reload
      expect(@team.name).to eq('something else')
    end

    it 'allows updates to settings.charity' do
      charity = Team.approved_charities.compact.sample
      patch "/v1/teams/#{@team.id}",
        params: { settings: {charity: charity} },
        headers: { 'User-ID': @owner.user_id }
      expect(response.status).to eq(200)
      @team.reload
      expect(@team.charity).to eq(charity)
    end

    it 'can change the visibility of a team by updating `settings["hidden"]`' do
      patch "/v1/teams/#{@team.id}",
        params: {settings: {hidden: true}},
        headers: admin_headers
      expect(response.status).to eq(200)
      expect(json['hidden']).to eq(true)
      @team.reload
      expect(@team.hidden).to eq(true)

      patch "/v1/teams/#{@team.id}",
        params: {settings: {hidden: false}},
        headers: admin_headers
      expect(response.status).to eq(200)
      expect(json['hidden']).to eq(false)
      @team.reload
      expect(@team.hidden).to eq(false)
    end

    it_behaves_like 'an image uploader for', :team, :logo, headers: admin_headers
    it_behaves_like 'an image uploader for', :team, :cover, headers: admin_headers
  end
end
