require 'rails_helper'

describe V1::TournamentEntriesController do
  it_behaves_like 'a #index API', { for: :tournament_entry }
  it_behaves_like 'a #destroy API', { for: :tournament_entry }

  describe 'GET /tournament_entries' do
    before :all do
      @team = FactoryGirl.create(:team)
      @league = FactoryGirl.create(:league_with_seasons, tournament_count: 2)
      @tournament = @league.tournaments.first
      @tournament.enter_team!(@team)
      FactoryGirl.create_list(:tournament_entry, 10, { tournament_id: @tournament.id })
    end

    it 'should only return entries for a tournament, given a tournament_id' do
      get '/v1/tournament_entries', params: { tournament_id: @tournament.id }
      expect(json.collect{|e| e['tournament_id']}.uniq).to eq([@tournament.id])
    end

    it 'should only return entries for an entrant, given an entrant_id' do
      get '/v1/tournament_entries', params: { entrant_id: @team.id }
      expect(json.collect{|e| e['entrant_id']}.uniq).to eq([@team.id])
    end

    it 'should only return entries for a team, given a team_id' do
      get '/v1/tournament_entries', params: { team_id: @team.id }
      expect(json.collect{|e| e['entrant_id']}.uniq).to eq([@team.id])
    end
  end

  describe 'POST /tournament_entries' do
    before :all do
      @team = FactoryGirl.create(:team_with_people)
      @owner = @team.members('owner').first.person
      @tournament = FactoryGirl.create(:tournament)
    end

    it 'requires a tournament' do
      post '/v1/tournament_entries',
        params: {team_id: @team.id},
        headers: {'User-ID': @owner.user_id}
      expect(response.status).to eq(400)
      expect(json['error']).to match(/Tournament must exist/)
    end

    # Note: we want to support other entities later on, which is why the
    # entrant association is polymorphic. But for now, all we need is teams.
    it 'requires an entrant' do
      post '/v1/tournament_entries',
        params: {tournament_id: @tournament.id},
        headers: {'User-ID': @owner.user_id}
      expect(response.status).to eq(400)
      expect(json['error']).to match(/Entrant must exist/)
    end

    it 'creates a valid TournamentEntry' do
      post '/v1/tournament_entries',
        params: {tournament_id: @tournament.id, team_id: @team.id},
        headers: {'User-ID': @owner.user_id}
      entry = TournamentEntry.find(json['id'])
      expect(entry).to be_valid
    end

    it 'allows setting flags' do
      post '/v1/tournament_entries',
        params: {flags: ['testing'], tournament_id: @tournament.id, team_id: @team.id},
        headers: {'User-ID': @owner.user_id}
      entry = TournamentEntry.find(json['id'])
      expect(entry.flags).to include('testing')
      expect(json['flags']).to include('testing')
    end
  end

  describe 'PATCH /tournament_entries' do
    before :all do
      @team = FactoryGirl.create(:team_with_people)
      @owner = @team.members('owner').first.person
      @entry = FactoryGirl.create(:tournament_entry, entrant: @team)
    end

    it 'updates flags' do
      patch "/v1/tournament_entries/#{@entry.id}",
        params: {flags: ['testing']},
        headers: {'User-ID': @owner.user_id}
      expect(response.status).to eq(200), "PATCH failed: "+JSON.pretty_generate(json)
      @entry.reload
      expect(@entry.flags).to include('testing')
      expect(json['flags']).to include('testing')
    end

    it 'can clear flags' do
      @entry.update!(flags: ['foo','bar'])
      patch "/v1/tournament_entries/#{@entry.id}",
        headers: {'User-ID': @owner.user_id},
        params: {"flags":[]}, as: :json
      expect(response.status).to eq(200), "PATCH failed: "+JSON.pretty_generate(json)
      expect(json['flags'].length).to eq(0)
      expect(@entry.reload.flags.length).to eq(0)
    end
  end
end
