require 'rails_helper'

describe V1::TournamentsController do
  it_behaves_like 'a CRUD API', { for: :tournament }, ->{{
    create: {name: Faker::Company.name, entrants_type: 'Team'},
    update: {name: Faker::Company.name}
  }}

  it_behaves_like 'an image uploader for', :tournament, :logo, headers: admin_headers
  it_behaves_like 'an image uploader for', :tournament, :cover, headers: admin_headers

  describe 'GET /tournaments' do
    before :each do
      @tournaments = FactoryGirl.create_list(:tournament, 3)
    end

    it 'returns tournaments that belong to game_ids if given' do
      game = FactoryGirl.create(:game)
      tournament = FactoryGirl.create(:tournament, game: game)
      get '/v1/tournaments',
        params: {game_ids: [game.id]}
      expect(response).to be_ok
      expect(json.collect{|t| t['id']}.sort).to eq([tournament.id])
    end
  end

  describe 'GET /tournaments/:id' do
    before :all do
      @tournament = FactoryGirl.create(:tournament)
    end

    it 'can look up a tournament by slug' do
      expect(@tournament.slug).to_not be_blank
      get "/v1/tournaments/#{@tournament.slug}"
      expect(response).to be_ok
      expect(json['id']).to eq(@tournament.id)
      expect(json['slug']).to eq(@tournament.slug)
    end
  end

  describe 'POST /tournaments' do
    it 'requires a name' do
      post '/v1/tournaments', params: {}, headers: admin_headers
      expect(json['errors']['name'].join).to match(/can't be blank/)
    end
  end

  describe 'PATCH /tournaments' do
    before :each do
      @tournament = FactoryGirl.create(:tournament)
    end

    it 'accepts nested attributes for stages via :stages' do
      stage = @tournament.stages.first
      patch "/v1/tournaments/#{@tournament.id}",
        params: { stages: [{ id: stage.id, name: 'Hermods Ride To Hel' }] },
        headers: admin_headers
      @tournament.reload
      expect(@tournament.stages.first.name).to eq('Hermods Ride To Hel')
    end
  end
end
