data "template_file" "edb_task_definition" {
  template = "${file("task-definitions/edb.json")}"
  vars {
    name = "edb"
    region = "us-west-2"
    image = "${replace(aws_ecr_repository.edb.repository_url, "https://", "")}:latest"
    db_url = "postgres://${aws_db_instance.default.username}:${aws_db_instance.default.password}@${aws_db_instance.default.endpoint}/${aws_db_instance.default.name}"
    container_port = 3000
    cpu = 512
    mem = 512
  }
}

resource "aws_iam_role" "task" {
  name = "ecs_${var.service_name}_task_role"
  assume_role_policy = <<EOF
{
  "Version": "2008-10-17",
  "Statement": [{
    "Action": "sts:AssumeRole",
    "Principal": {
      "Service": "events.amazonaws.com"
    },
    "Effect": "Allow"
  }, {
    "Action": "sts:AssumeRole",
    "Principal": {
      "Service": "ecs-tasks.amazonaws.com"
    },
    "Effect": "Allow"
  }]
}
EOF
}

resource "aws_iam_role_policy" "task" {
  name = "ecs_${var.service_name}_task_role"
  role = "${aws_iam_role.task.id}"
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": ["kinesis:*"],
      "Resource": ["*"]
    },
    {
      "Effect": "Allow",
      "Action": ["s3:*"],
      "Resource": ["arn:aws:s3:::edb-assets.${var.zone}/*"]
    }
  ]
}
EOF
}

resource "aws_ecs_task_definition" "edb_task" {
  family = "edb"
  container_definitions = "${data.template_file.edb_task_definition.rendered}"
  task_role_arn = "${aws_iam_role.task.arn}"
}

module "service" {
  source = "git::ssh://git@git-aws.internal.justin.tv/esports/terraform.git//modules/service?ref=2017-05-19"
  name = "edb"
  prefix = "edb"
  container_port = "3000"
  alb = true
  rails_json = true
  env = "${var.service_env}"
  desired_count = 2
  cluster = "tourney"
  certificate_arn = "${module.common.service_certificate_external}"
  task_definition = "${aws_ecs_task_definition.edb_task.arn}"
  vpc_id = "${lookup(data.terraform_remote_state.aws.ecs_cluster_tourney, "vpc_id")}"
  subnet_ids = "${split(",", lookup(data.terraform_remote_state.aws.ecs_cluster_tourney, "subnet_ids"))}"
  security_group_id = "${lookup(data.terraform_remote_state.aws.ecs_cluster_tourney, "security_group_id")}"
  internal_zone_id = "${data.terraform_remote_state.aws.route53_zone_services_internal}"
  external_zone_id = "${data.terraform_remote_state.aws.route53_zone_services_external}"
  alarm_sns_topic_arn = "${data.terraform_remote_state.aws.alarm_sns_topic_arn}"
}
