resource "aws_s3_bucket" "assets" {
  bucket = "edb-assets.staging.esports.internal.justin.tv"
  acl = "public-read"
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "AddPerm",
      "Effect": "Allow",
      "Principal": "*",
      "Action": "s3:GetObject",
      "Resource": "arn:aws:s3:::edb-assets.staging.esports.internal.justin.tv/*"
    }
  ]
}
EOF
  website {
    index_document = "index.html"
    error_document = "error.html"
  }
}

data "aws_acm_certificate" "assets" {
  domain = "*.staging.esports.internal.justin.tv"
  statuses = ["ISSUED"]
  provider = "aws.us-east-1"
}

resource "aws_cloudfront_distribution" "assets" {
  origin {
    domain_name = "${aws_s3_bucket.assets.bucket_domain_name}"
    origin_id   = "myS3Origin"

    # s3_origin_config {
    #   origin_access_identity = "origin-access-identity/cloudfront/ABCDEFG1234567"
    # }
  }

  enabled             = true
  default_root_object = "index.html"

  # logging_config {
  #   include_cookies = false
  #   bucket          = "mylogs.s3.amazonaws.com"
  #   prefix          = "myprefix"
  # }

  aliases = ["edb-assets.staging.esports.internal.justin.tv"]

  default_cache_behavior {
    allowed_methods  = ["DELETE", "GET", "HEAD", "OPTIONS", "PATCH", "POST", "PUT"]
    cached_methods   = ["GET", "HEAD"]
    target_origin_id = "myS3Origin"

    forwarded_values {
      query_string = false

      cookies {
        forward = "none"
      }
    }

    viewer_protocol_policy = "allow-all"
    min_ttl                = 0
    default_ttl            = 3600
    max_ttl                = 86400
  }

  price_class = "PriceClass_200"

  # Why is this required?
  restrictions {
    geo_restriction {
      restriction_type = "blacklist"
      locations        = ["ML"]
    }
  }

  tags {
    Environment = "stage"
    Service = "esports/${var.service_name}"
    Owner = "marian@twitch.tv"
  }

  viewer_certificate {
    acm_certificate_arn = "${data.aws_acm_certificate.assets.arn}"
    ssl_support_method = "sni-only"
    minimum_protocol_version = "TLSv1"
  }
}

data "aws_route53_zone" "internal" {
  name = "esports.internal.justin.tv."
  private_zone = true
}

resource "aws_route53_record" "assets" {
  zone_id = "${data.aws_route53_zone.internal.zone_id}"
  name = "edb-assets.staging.${data.aws_route53_zone.internal.name}"
  type = "A"
  alias = {
    name = "${aws_cloudfront_distribution.assets.domain_name}"
    zone_id = "${aws_cloudfront_distribution.assets.hosted_zone_id}"
    evaluate_target_health = false
  }
}
