import * as React from 'react';
import { debounce } from 'lodash-es';

import { StreamerPreview } from 'ui/components/streamer-preview';

require('./styles.scss');

const CHAT_THRESHOLD_WIDTH = 1100;
const MAX_STREAM_WIDTH = 1300;
const CHAT_WIDTH = 340;
const TARGET_ASPECT_RATIO = 16 / 9;

export class LiveStream extends React.Component<any, any> {
  public boundResizeHandler: (event: any) => {};
  public constructor(props: any) {
    super(props);
    this.state = {
      windowSize: {
        width: window.innerWidth,
        height: window.innerHeight
      }
    };
    this.boundResizeHandler = debounce(this.updateWindowSize.bind(this), 500);
  }

  public componentDidMount() {
    this.updateWindowSize();
    window.addEventListener('resize', this.boundResizeHandler);
  }

  public componentWillUnmount() {
    window.removeEventListener('resize', this.boundResizeHandler);
  }

  public render() {
    const { contestant } = this.props;
    const streamSrc = `https://player.twitch.tv/?channel=${contestant.name}`;
    const chatSrc = `https://www.twitch.tv/${contestant.name}/chat`;
    const hasChat = this.state.windowSize.width >= CHAT_THRESHOLD_WIDTH;
    const logo = require('common/assets/ow-logo.png') as string;
    const logoStyle = { backgroundImage: `url(${logo})` };
    const imageSrc = require(`common/assets/streamers/${contestant.image}`) as string;
    const imageStyle = { backgroundImage: `url(${imageSrc})` };

    const { width, height } = this.getTakeoverStyles(hasChat);

    let chat = hasChat
      ? (
        <iframe
          className="live-stream__chat"
          frameBorder="0"
          scrolling="yes"
          id="chat_embed"
          src={chatSrc}
          height="100%"
          width={CHAT_WIDTH}
        />
      )
      : undefined;

    const containerWidth = hasChat ? width + CHAT_WIDTH : width;
    return (
      <div className=" section-container  section-container--hero">
        <div className="live-stream__container">
          <div className="live-stream__logo" style={logoStyle} />
          <div className="live-stream__title">Château Showdown</div>
          <div className="live-stream__contestant">
            <div className="live-stream__preview" style={imageStyle}/>
            <div className="live-stream__name">{contestant.name}</div>
            <a href={`https://twitch.tv/${contestant.name}`}>
              <div className="live-stream__button">Watch On Twitch</div>
            </a>
          </div>
          <div className="live-stream__contents" style={{ height }}>
            <div className="live-stream__stream" style={{ width, height }}>
              <iframe
                src={streamSrc}
                width="100%"
                height="100%"
                frameBorder="0"
                scrolling="no"
                allowFullScreen={true}
              />
            </div>
            {chat}
          </div>
        </div>
      </div>
    );
  }


  private getTakeoverStyles(hasChat: boolean) {
    const { windowSize } = this.state;
    const SCROLLBAR_OFFSET = windowSize.width > 768 ? 15 : 0;
    const useableHeight = windowSize.height;
    let useableWidth = windowSize.width - SCROLLBAR_OFFSET;
    useableWidth = useableWidth > MAX_STREAM_WIDTH ? MAX_STREAM_WIDTH : useableWidth;
    if (hasChat) {
      useableWidth -= CHAT_WIDTH;
    }

    let width = useableWidth;
    let height: number;
    let idealHeight = width / TARGET_ASPECT_RATIO;
    if (idealHeight <= useableHeight) {
      height = idealHeight;
    } else {
      height = useableHeight;
      width = height * TARGET_ASPECT_RATIO;
    }

    return {
      width: Math.floor(width),
      height: Math.floor(height)
    };
  }

  private updateWindowSize() {
    this.setState({
      windowSize: {
        width: window.innerWidth,
        height: window.innerHeight
      }
    });
  }
}
