import { CHANNEL, CHANNEL_ID, GAME } from '../common/constants';
import { config } from 'config';
require('isomorphic-fetch');

import { StreamProps, VideoProps } from 'state/stream/model';

import { Action } from './action-type';

export const SET_LIVE = 'SET_LIVE';
export interface SetLiveAction extends Action<typeof SET_LIVE, boolean> {}

export function createSetLiveAction(status: boolean): SetLiveAction {
  return {
    type: SET_LIVE,
    payload: status
  };
}

export const SET_STREAMS = 'SET_STREAMS';
export interface SetStreamsAction extends Action<typeof SET_STREAMS, StreamProps[]> {}

export function createSetStreamAction(streams: StreamProps[]): SetStreamsAction {
  return {
    type: SET_STREAMS,
    payload: streams
  };
}

export const SET_VIDEOS = 'SET_VIDEOS';
export interface SetVideosAction extends Action<typeof SET_VIDEOS, VideoProps[]> {}

export function createSetVideosAction(videos: VideoProps[]): SetVideosAction {
  return {
    type: SET_VIDEOS,
    payload: videos
  };
}

function fetchWithHeaders(url: string) {
  return fetch(new Request(url, {
    headers: new Headers({
      Accept: 'application/vnd.twitchtv.v5+json',
      'Client-ID': config.twitch_client,
    })
  }));
}

export function fetchVideos() {
  let url = `https://api.twitch.tv/kraken/channels/${CHANNEL_ID}/videos?limit=4`;
  return (dispatch: any) => {
    return fetchWithHeaders(url)
      .then((response: any) => response.json())
      .then((data: any) => {
        if (data.error) {
           dispatch(createSetVideosAction([]));
        } else {
          const videoData = data.videos.map((video: any) => parseVideo(video));
          dispatch(createSetVideosAction(videoData));
        }
      });
  };
}

function parseVideo(video: any) {
  return {
    id: video._id,
    title: video.title,
    preview: video.preview.large,
    views: video.views,
    link: video.url,
  };
}

export function fetchLive() {
  let url = `https://api.twitch.tv/kraken/streams/${CHANNEL_ID}`;
  return (dispatch: any) => {
    return fetchWithHeaders(url)
      .then((response: any) => response.json())
      .then((data: any) => {
        if (data.error) {
          dispatch(createSetLiveAction(false));
        } else {
          dispatch(createSetLiveAction(data.stream !== null));
        }
      });
  };
}

export function fetchStreams() {
  let channels = require('common/constants/playerIds.json') as string[];
  let idsString = channels.reduce((memo, obj) => `${memo},${obj}`, '').substring(1);
  let url = `https://api.twitch.tv/kraken/streams/?channel=${idsString}&game=${GAME}&limit=4`;
  return (dispatch: any) => {
    return fetchWithHeaders(url)
      .then((response: any) => response.json())
      .then((data: any) => {
        if (data.error) {
          dispatch(createSetStreamAction([]));
        } else {
          let streamData = data.streams.map((stream: any) => parseStream(stream));
          dispatch(createSetStreamAction(streamData));
        }
      });
  };
}

function parseStream(stream: any) {
  return {
    title: stream.channel.display_name,
    preview: stream.preview.large,
    viewers: stream.viewers,
    link: stream.channel.url,
  };
}

export type StreamActions = SetLiveAction | SetStreamsAction | SetVideosAction;
