import * as _ from 'lodash';
import * as React from 'react';
import * as moment from 'moment';
import { connect } from 'react-redux';
import { NavLink, Link, withRouter } from 'react-router-dom';
import { bindActionCreators, Dispatch } from 'redux';
import {
  denormalize,
  SeriesSchema,
  ScheduledEvent,
  SeriesModel,
  getTeams,
  TournamentStageSchema,
  getTournaments,
  getSeasons,
  SeasonModel
 } from 'tourney-sdk-react';

import { fetchSchedule, fetchLive } from 'actions';
import { AppModel } from 'state/model';
import { Navigation } from 'ui/components/navigation';
import { Footer } from 'ui/components/footer';
import { TickerNav } from 'ui/components/ticker-nav';
import { TOURNAMENTS, ALL_TOURNAMENTS, SEASON_NAME, SCHEDULE_INTERVAL } from 'common/constants';

require('common/styles/app.scss');

class AppComponent extends React.Component<any, any> {
  public static mapStateToProps(state: AppModel, ownProps: any) {
    const orderedSeries = denormalize(state.schedule.items, [ScheduledEvent], state.tourney);
    return {
      orderedSeries
    };
  }

  public static mapDispatchToProps(dispatch: Dispatch<AppModel>, ownProps: AppModel) {
    return {
      dispatch
    };
  }
  constructor(...args: any[]) {
    super(args[0]);
  }

  public componentDidMount() {
    this.props.dispatch(fetchLive());
    this.props.dispatch(getTournaments());
    this.props.dispatch(getSeasons())
      .then((seasons: any) => {
        const s = _.find(seasons, (season: SeasonModel) => season.name === SEASON_NAME );
        this.fetchScheduleAndTeams(s);
        setInterval(this.fetchScheduleBySeason.bind(this, s), SCHEDULE_INTERVAL * 1000);
      });
  }

  public render() {
    const classes = ['app-container'];

    return (
      <div className={classes.join(' ')}>
        <TickerNav />
        <Navigation />
        {this.props.children}
        <Footer />
      </div>
    );
  }

  private fetchScheduleBySeason(season: SeasonModel) {
    const today = moment().tz('America/Los_Angeles');
    const initFrom = today.clone().subtract(1, 'year').startOf('day').toISOString();
    const initTo = today.clone().add(1, 'year').endOf('day').toISOString();
    return this.props.dispatch(fetchSchedule(initFrom, initTo, season.id));
  }

  private fetchScheduleAndTeams(season: SeasonModel) {
    this.fetchScheduleBySeason(season)
      .then(() => {
        // filter for only series, then find unique team ids.
        const teamIds = this.props.orderedSeries
          .filter((item: any) => item.opponents)
          .reduce((memo: any[], series: any) => {
            return [
              ...memo,
              ...series.opponents.filter((opp: any) => opp.contender_type === 'Team').map((opp: any) => opp.contender_id)
            ];
          }, []);
        return this.props.dispatch(getTeams({ ids: _.uniq(teamIds) }));
      });
  }
}

export const App = withRouter(connect(AppComponent.mapStateToProps, AppComponent.mapDispatchToProps)(AppComponent));
