import * as _ from 'lodash';
import * as React from 'react';
import {
  TournamentStageModel
} from 'tourney-sdk-react';

import { BracketType, BracketRound, BracketSeries } from 'common/constants';
require('./styles.scss');

export interface BracketPreviewProps {
  tournamentStage: TournamentStageModel;
  format: BracketType[];
  contentSizePct: number;
  leftOffsetPct: number;
  updateLeftPct: (pct: number) => {};
}

export interface BracketPreviewState {
  dragging: boolean;
  draggingOffsetPct: number;
  leftPct: number;
}

export class BracketPreview extends React.Component<BracketPreviewProps, BracketPreviewState> {
  private selectedBox: HTMLDivElement;
  private previewContainer: HTMLDivElement;
  private boundHandleDragStart: (event: MouseEvent) => {};
  private boundHandleDrag: (event: MouseEvent) => {};
  private boundHandleDragEnd: () => {};
  private throttledParentUpdate: (pct: number) => {};
  public constructor(props: BracketPreviewProps) {
    super(props);
    this.state = {
      dragging: false,
      draggingOffsetPct: undefined,
      leftPct: undefined
    };
    this.boundHandleDragStart = this.handleDragStart.bind(this);
    this.boundHandleDrag = this.handleDrag.bind(this);
    this.boundHandleDragEnd = this.handleDragEnd.bind(this);
    this.throttledParentUpdate = _.throttle(this.props.updateLeftPct, 50);
  }

  public componentDidMount() {
    this.selectedBox.addEventListener('mousedown', this.boundHandleDragStart);
    this.selectedBox.addEventListener('touchstart', this.boundHandleDragStart);
    this.setState({
      leftPct: this.props.leftOffsetPct
    });
  }

  public componentWillUnmount() {
    this.selectedBox.removeEventListener('mousedown', this.boundHandleDragStart);
    this.selectedBox.removeEventListener('touchstart', this.boundHandleDragStart);
  }

  public componentDidUpdate(prevProps: BracketPreviewProps) {
    const { leftOffsetPct } = this.props;
    if (!Number.isNaN(leftOffsetPct) && prevProps.leftOffsetPct !== leftOffsetPct) {
      this.setState({
        leftPct: this.props.leftOffsetPct
      });
    }
  }

  public render() {
    const { format, contentSizePct, leftOffsetPct } = this.props;
    const { leftPct } = this.state;

    const bracketTypes = format.map(this.renderBracketType.bind(this));
    const selectedStyles = {
      width: `${contentSizePct * 100}%`,
      left: `${leftPct * 100}%`
    };
    const setupPreviewContainerRef = (previewContainer: HTMLDivElement) => {
      this.previewContainer = previewContainer;
    };
    const setupSelectedBoxRef = (selectedBox: HTMLDivElement) => {
      this.selectedBox = selectedBox;
    };
    let selectedBoxClasses = ['bracket-preview__selected-box'];
    if (this.state.dragging) {
      selectedBoxClasses = [...selectedBoxClasses, 'bracket-preview__selected-box--dragging'];
    }
    return (
      <div className="bracket-preview__container" ref={setupPreviewContainerRef}>
        {bracketTypes}
        <div className={selectedBoxClasses.join(' ')} style={selectedStyles} ref={setupSelectedBoxRef}/>
      </div>
    );
  }

  private handleDragStart(event: any) {
    document.addEventListener('mousemove', this.boundHandleDrag);
    document.addEventListener('touchmove', this.boundHandleDrag);
    document.addEventListener('mouseup', this.boundHandleDragEnd);
    document.addEventListener('touchend', this.boundHandleDragEnd);
    const rect = this.previewContainer.getBoundingClientRect();
    const x = event.touches ? event.touches[0].clientX : event.clientX;
    const pct = (x - rect.left) / rect.width;
    this.setState({
      dragging: true,
      draggingOffsetPct: pct - this.state.leftPct,
    });
  }

  private handleDrag(event: any) {
    const rect = this.previewContainer.getBoundingClientRect();
    const x = event.touches ? event.touches[0].clientX : event.clientX;
    const locPct = (x - rect.left) / rect.width;
    let newLeftPct = locPct - this.state.draggingOffsetPct;
    if (newLeftPct < 0) {
      newLeftPct = 0;
    } else if (newLeftPct + this.props.contentSizePct > 1) {
      newLeftPct = 1 - this.props.contentSizePct;
    }
    this.setState({
      leftPct: newLeftPct
    });
    this.throttledParentUpdate(newLeftPct);
  }

  private handleDragEnd() {
    document.removeEventListener('mousemove', this.boundHandleDrag);
    document.removeEventListener('touchmove', this.boundHandleDrag);
    document.removeEventListener('mouseup', this.boundHandleDragEnd);
    document.removeEventListener('touchend', this.boundHandleDragEnd);
    this.setState({
      dragging: false,
      draggingOffsetPct: undefined,
    });
  }

  private renderBracketType(bracketType: BracketType): JSX.Element {
    const rounds = bracketType.rounds.map((round: BracketRound, roundIdx: number) => {
      return this.renderRound(round, bracketType, roundIdx);
    });
    return (
      <div className="bracket-preview__type" key={bracketType.name}>
        {rounds}
      </div>
    );
  }

  private renderRound(round: BracketRound, bracketType: BracketType, roundIdx: number) {
    const series = round.series.map((bracketSeries: BracketSeries, seriesIdx: number) => {
      return this.renderSeries(bracketSeries, seriesIdx);
    });
    return (
      <div className="bracket-preview__round" key={roundIdx}>
        {series}
      </div>
    );
  }

  private renderSeries(series: BracketSeries, seriesIdx: number) {
    let classes = ['bracket-preview__series', 'noselect'];
    if (series.ident === undefined) {
      classes = [...classes, 'bracket-preview__series--empty'];
    }
    return (
      <div className={classes.join(' ')} key={seriesIdx}/>
    );
  }
}
