import * as _ from 'lodash';
import * as React from 'react';
import { connect } from 'react-redux';
import { Link } from 'react-router-dom';
import { bindActionCreators, Dispatch } from 'redux';
import {
  denormalize,
  getSeries,
  getTeams,
  TournamentStageSchema,
  TournamentStageModel,
  ScheduledEvent,
  SeriesModel,
  SeriesSchema,
  TeamSchema,
  TeamModel
} from 'tourney-sdk-react';

import { BracketType, BracketRound, BracketSeries } from 'common/constants';
import { removeDisqualified } from 'common/helpers';
import { AppModel } from 'state/model';
import { TeamLink } from 'ui/components/team-link';
import { ScheduleSeries } from 'ui/components/schedule-series';
import { TournamentHeader } from 'ui/components/tournament-header';
import { BracketPreview, BracketPreviewProps } from 'ui/components/bracket-preview';

require('./styles.scss');

const BRACKET_ROUND_WIDTH = 325;
const MARGIN_RIGHT = 75;
const SIDE_PADDING = 20;

export interface BracketProps {
  tournamentStage: TournamentStageModel;
  series: {[key: string]: SeriesModel};
  format: BracketType[];
  dispatch?: any;
}

export type ScrollDirections = 'left' | 'right';

export class BracketsComponent extends React.Component<any, any> {
  public static mapStateToProps(state: AppModel, ownProps: any): any {
    const orderedSeries = removeDisqualified(denormalize(state.schedule.items, [ScheduledEvent], state.tourney));
    // const orderedSeries = denormalize(_.keys(state.tourney.series), [SeriesSchema], state.tourney);
    const series = _.keyBy(orderedSeries.filter((item: SeriesModel) => {
      return item.tournament_stage_id === ownProps.tournamentStage.id;
    }) as SeriesModel[], 'ident');
    const loading = _.isEmpty(series);
    return {
      ...ownProps,
      series,
      loading,
    };
  }

  public container: HTMLDivElement;
  public contents: HTMLDivElement;
  private contentsScrolledAction: (event: Event) => {};
  public constructor(props: any) {
    super(props);
    this.state = {
      containerWidth: undefined,
      contentsWidth: undefined,
      contentsLeft: undefined,
    };

    this.contentsScrolledAction = _.debounce(this.contentsScrolled.bind(this), 100);
  }

  public componentDidMount() {
    // scroll listener
    this.container.addEventListener('scroll', this.contentsScrolledAction);
    window.addEventListener('resize', this.contentsScrolledAction);
    this.setState({
      containerWidth: this.container.clientWidth,
    });
    setTimeout(() => {
      this.updateContentsState();
    }, 0);
  }

  public componentWillUnmount() {
    this.container.removeEventListener('scroll', this.contentsScrolledAction);
    window.removeEventListener('resize', this.contentsScrolledAction);
  }

  public render() {
    const { loading, tournamentStage, format } = this.props;
    const { contentsWidth, containerWidth, contentsLeft } = this.state;
    let content = <div>Loading</div>;
    if (!loading) {
      content = this.renderSingleEliminationBracket();
    }
    const setupContainerRef = (container: HTMLDivElement) => {
      this.container = container;
    };
    const scrollLeft = () => this.scrollBrackets('left');
    const scrollRight = () => this.scrollBrackets('right');
    const bracketPreviewProps: BracketPreviewProps = {
      tournamentStage,
      format,
      contentSizePct: containerWidth / contentsWidth,
      leftOffsetPct: contentsLeft / contentsWidth,
      updateLeftPct: this.scrollBracketsToPct.bind(this),
    };

    let previewHidden;
    let scrollContainerClasses = ['bracket-series__scroll-container'];
    if (bracketPreviewProps.contentSizePct >= 1 || loading) {
      previewHidden = 'hidden';
      scrollContainerClasses.push('bracket-series__scroll-container--unfilled');
    }
    return (
      <div className="bracket-series__container" key={tournamentStage.id}>
        <TournamentHeader tournament={tournamentStage.tournament}>
          <div className={['bracket-series__controls-container', previewHidden].join(' ')}>
            <div className="bracket-series__controls-arrow" onClick={scrollLeft}>
              <i className="fa fa-chevron-left" />
            </div>
            <BracketPreview {...bracketPreviewProps}/>
            <div className="bracket-series__controls-arrow" onClick={scrollRight}>
              <i className="fa fa-chevron-right" />
            </div>
          </div>
        </TournamentHeader>
        <div className={scrollContainerClasses.join(' ')} ref={setupContainerRef}>
          {content}
        </div>
      </div>
    );
  }

  private scrollBracketsToPct(leftPct: number, smooth = false) {
    const scrollSettings: ScrollToOptions = {
      left: leftPct * this.state.contentsWidth,
    };
    if (smooth) {
       scrollSettings.behavior = 'smooth';
    }
    this.container.scrollTo(scrollSettings);
  }

  private scrollBrackets(direction: 'left' | 'right') {
    let newLeft;
    if (direction === 'left') {
      newLeft = this.state.contentsLeft - this.state.containerWidth;
    } else {
      newLeft = this.state.contentsLeft + this.state.containerWidth;
    }

    const max = this.state.contentsWidth - this.state.containerWidth;
    if (newLeft < 0) {
      newLeft = 0;
    } else if (newLeft > max) {
      newLeft = max;
    }

    this.container.scrollTo({ left: newLeft, behavior: 'smooth' });
  }

  private renderSingleEliminationBracket() {
    const { format, tournamentStage } = this.props;
    const sections = format.map((bracketType: BracketType) => {
      const rounds = bracketType.rounds.map((bracketRound: BracketRound, idx: number) => {
        return this.renderRound(bracketRound, bracketType, idx);
      });
      return (
        <div className="bracket-overview__container" key={bracketType.name}>
          <div className="bracket-overview__round-list">
            {rounds}
          </div>
        </div>
      );
    });

    const maxRounds = format.map((bracketType: BracketType) => {
      return bracketType.rounds.length;
    });
    const width = Math.max(...maxRounds) * (BRACKET_ROUND_WIDTH + MARGIN_RIGHT) - MARGIN_RIGHT + (SIDE_PADDING * 2);
    const styling = { width: `${width}px` };
    const setupContentsRef = (contents: HTMLDivElement) => {
      this.contents = contents;
    };
    const { contentLeft } = this.state;
    return (
      <div className="bracket-series__contents" ref={setupContentsRef} style={styling} key={tournamentStage.id}>
        {sections}
      </div>
    );
  }

  private renderRound(bracketRound: BracketRound, bracketType: BracketType, idx: number) {
    const series = bracketRound.series.map((bracketSeries: BracketSeries, seriesIdx: number) => {
      return this.renderSeriesItem(bracketSeries, bracketRound, bracketType, seriesIdx);
    });
    return (
      <div className="bracket-round" key={`${bracketType.name}-${bracketRound.title}-${idx}`}>
        <div className="bracket-round__title">{bracketRound.title}</div>
        <ul className="bracket-round__list">
          {series}
        </ul>
      </div>
    );
  }

  private renderSeriesItem(bracketSeries: BracketSeries, bracketRound: BracketRound, bracketType: BracketType, idx: number) {
    const series = this.props.series[bracketSeries.ident];
    let classNames = [`bracket-item__height--${bracketType.height}`, `bracket-item__size--${bracketRound.size}`];
    if (bracketSeries.connection) {
      classNames = [...classNames, `schedule-series__container--${bracketSeries.connection}`];
    }

    if (!series) {
      return <div className="schedule-series__filler" key={idx}/>;
    }
    return <ScheduleSeries series={series} showDate={true} classNames={classNames} key={bracketSeries.ident} />;
  }

  private updateContentsState() {
    this.setState({
      containerWidth: this.container.clientWidth,
      contentsLeft: this.container.scrollLeft,
      contentsWidth: this.contents.clientWidth
    });
  }

  private contentsScrolled(event: Event) {
    this.updateContentsState();
  }
}

export const Brackets = connect(BracketsComponent.mapStateToProps)(BracketsComponent);
