import * as _ from 'lodash';
import * as React from 'react';
import { Link } from 'react-router-dom';

import { Hero } from 'ui/components/hero';
import { HeroSlider } from 'ui/components/hero-slider';
import { AppModel } from 'state/model';
import {
  CHANNEL,
  DESKTOP_NAVIGATION_HEIGHT,
  MOBILE_NAVIGATION_HEIGHT
} from 'common/constants';

require('./styles.scss');

const CHAT_THRESHOLD_WIDTH = 1100;
const CHAT_WIDTH = 340;
const TARGET_ASPECT_RATIO = 16 / 9;

export interface LiveTakeoverState {
  windowSize: {
    width: number,
    height: number
  };
}

export class LiveTakeover extends React.Component<{}, LiveTakeoverState> {
  public boundResizeHandler: (event: any) => {};
  public constructor(props: any) {
    super(props);
    this.state = {
      windowSize: {
        width: window.innerWidth,
        height: window.innerHeight
      }
    };
    this.boundResizeHandler = _.debounce(this.updateWindowSize.bind(this), 500);
  }

  public componentDidMount() {
    this.updateWindowSize();
    window.addEventListener('resize', this.boundResizeHandler);
  }

  public componentWillUnmount() {
    window.removeEventListener('resize', this.boundResizeHandler);
  }

  public render() {
    const containerClasses = ['live-takeover'];
    const streamSrc = `https://player.twitch.tv/?channel=${CHANNEL}`;
    const chatSrc = `https://www.twitch.tv/${CHANNEL}/chat`;
    const hasChat = this.state.windowSize.width >= CHAT_THRESHOLD_WIDTH;
    const { width, height } = this.getTakeoverStyles(hasChat);

    let chat = hasChat
      ? (
        <iframe
          className="live-takeover__chat"
          frameBorder="0"
          scrolling="yes"
          id="chat_embed"
          src={chatSrc}
          height="100%"
          width={CHAT_WIDTH}
        />
      )
      : undefined;

    const containerWidth = hasChat ? width + CHAT_WIDTH : width;
    return (
      <div className={containerClasses.join(' ')} style={{ height }}>
        <div className="live-takeover__stream" style={{ width, height }}>
          <iframe
            src={streamSrc}
            width="100%"
            height="100%"
            frameBorder="0"
            scrolling="no"
            allowFullScreen={true}
          />
        </div>
        {chat}
      </div>
    );
  }

  private getTakeoverStyles(hasChat: boolean) {
    const { windowSize } = this.state;
    const SCROLLBAR_OFFSET = windowSize.width > 768 ? 10 : 0;
    const useableHeight = windowSize.height - DESKTOP_NAVIGATION_HEIGHT;
    let useableWidth = windowSize.width - SCROLLBAR_OFFSET;
    if (hasChat) {
      useableWidth -= CHAT_WIDTH;
    }

    let width = useableWidth;
    let height: number;
    let idealHeight = width / TARGET_ASPECT_RATIO;
    if (idealHeight <= useableHeight) {
      height = idealHeight;
    } else {
      height = useableHeight;
      width = height * TARGET_ASPECT_RATIO;
    }

    return {
      width: Math.floor(width),
      height: Math.floor(height)
    };
  }

  private updateWindowSize() {
    this.setState({
      windowSize: {
        width: window.innerWidth,
        height: window.innerHeight
      }
    });
  }
}
