import * as _ from 'lodash';
import * as React from 'react';
import { connect } from 'react-redux';
import { NavLink, Link, withRouter } from 'react-router-dom';

import { SECTIONS, SOCIALS, SocialLink, REGISTER_URL, CHANNEL, TICKER_THRESHOLD } from 'common/constants';
import { AppModel } from 'state/model';

require('./styles.scss');

export interface Section {
  title: string;
  url: string;
  activeUrls?: RegExp[];
}

export interface NavigationState {
  mobileNavOpen: boolean;
  scrolledThreshold: boolean;
}

export class NavigationComponent extends React.Component<any, NavigationState> {
  public static mapStateToProps(state: AppModel) {
    return {
      live: state.stream.live
    };
  }

  public boundScrollHandler: (event: any) => {};
  public constructor(props: any) {
    super(props);

    this.state = {
      mobileNavOpen: false,
      scrolledThreshold: false
    };
    this.boundScrollHandler = this.handleScroll.bind(this);
  }

  public componentDidMount() {
    window.addEventListener('scroll', this.boundScrollHandler);
  }

  public componentWillUnmount() {
    window.removeEventListener('scroll', this.boundScrollHandler);
  }

  public render() {
    const headerLogo = require('common/assets/logo_s4.svg') as string;
    const rocketLeagueText = require('common/assets/text_RL.svg') as string;
    const toggleAction = this.toggleMobileNav.bind(this);
    const closeNav = this.closeMobileNav.bind(this);

    const sections: JSX.Element[] = SECTIONS.map((section) => {
      let isActive = (s: Section) => {
        return (match: any, location: any) => {
          if (location.pathname === s.url) {
            return true;
          }

          // check RegEx if it has them
          return s.activeUrls && _.some(s.activeUrls.map((url: RegExp) => location.pathname.match(url)));
        };
      };
      return (
        <li className="nav-sections__item" key={section.title}>
          <NavLink to={section.url} isActive={isActive(section)} onClick={closeNav}>
            {section.title}
          </NavLink>
        </li>
      );
    });

    let classes = ['navigation__contents'];
    if (this.state.mobileNavOpen) {
      classes = [...classes, 'navigation__contents--open'];
    }

    if (this.state.scrolledThreshold) {
      classes = [...classes, 'scrolled'];
    }

    return (
      <div className="navigation">
        {/* This is the fixed Mobile Nav that lives at the top */}
        <div className="mobile-nav">
          <div
            className="mobile-nav__toggle"
            onClick={toggleAction}
          >
            <i className="fa fa-bars"/>
          </div>
          <img className="mobile-nav__logo" src={headerLogo}/>
          <div className="mobile-nav__twitch">
            <i className="fa fa-twitch" />
          </div>
        </div>

        {/* This is the main dekstop nav and mobile slide out */}
        <div className={classes.join(' ')}>
          <div className="navigation__container">
            <div className="nav-logo">
              {this.renderStreamStatus()}
              <Link to="/">
                <img className="nav-logo__logo" src={headerLogo}/>
                <img className="nav-logo__rl" src={rocketLeagueText}/>
                <div className="nav-logo__subtitle">
                  Championship<br/>
                  Series Season 4
                </div>
              </Link>
            </div>
            <ul className="nav-sections">
              {sections}
            </ul>
            <div className="navigation__socials">
              {this.renderSocialIcons()}
            </div>
          </div>
        </div>
      </div>
    );
  }

  private renderSocialIcons(): JSX.Element[] {
    return SOCIALS.map((socialLink: SocialLink) => <a href={socialLink.url} key={socialLink.label}><i className={socialLink.icon}/></a>);
  }

  private renderStreamStatus() {
    let classes = ['nav-live'];
    let text = 'Offline';

    if (this.props.live) {
      classes.push('nav-live--online');
      text = 'Live';
    }

    return (
      <div className={classes.join(' ')}>
        <a href={`https://twitch.tv/${CHANNEL}`}>
          <div className="nav-live__dot"/>
          <div className="nav-live__text">
            {text}
          </div>
        </a>
      </div>
    );
  }

  private handleScroll() {
    const pastThreshold = window.scrollY > TICKER_THRESHOLD;
    if (this.state.scrolledThreshold !== pastThreshold) {
      this.setState({
        scrolledThreshold: pastThreshold
      });
    }
  }

  private toggleMobileNav() {
    this.setState({
      mobileNavOpen: !this.state.mobileNavOpen
    });
  }

  private closeMobileNav() {
    this.setState({
      mobileNavOpen: false
    });
  }
}

export const Navigation = withRouter(connect(NavigationComponent.mapStateToProps)(NavigationComponent));
