import * as _ from 'lodash';
import * as React from 'react';
import * as moment from 'moment';
import { Link } from 'react-router-dom';
import {
  denormalize,
  ArticleSchema,
  ArticleModel,
  SeriesModel
} from 'tourney-sdk-react';

import { TeamLink } from 'ui/components/team-link';
import { createFillerOpponents } from 'common/helpers';

require('./styles.scss');

interface ScheduleSeriesProps {
  series: SeriesModel;
  showDate?: boolean;
  classNames?: string[];
}

export class ScheduleSeries extends React.Component<any, any> {
  constructor(props: any) {
    super(props);
  }

  public render() {
    const { series, classNames } = this.props;
    let content;
    // check if anyone has any games won
    const scoreArr = series.opponents.map((opp: any) => opp.score);
    if (_.some(scoreArr)) {
      content = this.renderScore();
    } else {
      content = this.renderScheduled();
    }
    let classes = ['schedule-series__container'];
    if (classNames) {
      classes = [...classes, ...classNames];
    }
    return (
      <div className={classes.join(' ')}>
        <div className="schedule-series__teams">
          {this.renderOpponents()}
        </div>
        {content}
      </div>
    );
  }

  private renderScore() {
    const { series } = this.props;
    const scores = series.opponents.map((opp: any) => {
      const classes = ['schedule-series__score', opp.winner_id ? 'schedule-series__score--winner' : ''];
      return (
        <div className={classes.join(' ')} key={opp.id}>
          {opp.score}
        </div>
      );
    });
    return (
      <div className="schedule-series__scores">
        {scores}
      </div>
    );
  }

  private renderScheduled() {
    const { series, showDate } = this.props;
    const westcoastTime = moment(series.scheduled_at).tz('America/Los_Angeles');
    // const westcoastTime = moment(series.scheduled_at).tz('America/New_York');
    const localTime = moment(series.scheduled_at).tz(moment.tz.guess());
    let dateElement = showDate
      ? <div className="schedule-series__date">{westcoastTime.format('MMM Do')}</div>
      : undefined;
    let localTimeElement: JSX.Element = undefined;
    if (westcoastTime.format('YYYY-MM-DD HH:mm') !== localTime.format('YYYY-MM-DD HH:mm')) {
      localTimeElement = (
        <div className="schedule-series__time-local">
          {localTime.format('LT')}<span>{localTime.format('z')}</span>
        </div>
      );
    }
    return (
      <div className="schedule-series__scheduled">
        <div className="schedule-series__time">
          {dateElement}
          {westcoastTime.format('LT')}<br/><span>{westcoastTime.format('z')}</span>
        </div>
        {localTimeElement}
      </div>
    );
  }

  private renderOpponents() {
    const { series } = this.props;
    const tempLogo = require('common/assets/no_logo.svg') as string;
    let opponents = series.opponents;
    if (opponents.length < 2) {
      const fillers = createFillerOpponents(2 - opponents.length);
      opponents = [...opponents, ...fillers];
    }
    return opponents.map((opp: any) => {
      let logo: string = opp.contender.logo.original ? opp.contender.logo.thumb : tempLogo;
      const logoElement = <div className="schedule-series__team-logo" style={{ backgroundImage: `url(${logo})` }}/>;
      const classes = ['schedule-series__team', opp.winner_id ? 'schedule-series__team--winner' : ''];
      return (
        <div className={classes.join(' ')} key={opp.id}>
          <TeamLink team={opp.contender}>
            {logoElement}
            {opp.contender.name}
          </TeamLink>
        </div>
      );
    });
  }
}
