import * as _ from 'lodash';
import * as React from 'react';
import * as moment from 'moment';
import { Link } from 'react-router-dom';
import { denormalize, ArticleSchema, ArticleModel } from 'tourney-sdk-react';

require('./styles.scss');

export interface ScheduleTimelineProps {
  availableDates: string[];
  setDate: (date: string) => {};
  selectedDate: string;
}

export interface ScheduleTimelineState {
  perPage: number;
  firstDate: string;
  scrollLocation: string;
}

export class ScheduleTimeline extends React.Component<ScheduleTimelineProps, ScheduleTimelineState> {
  public boundResizeHandler: (event: any) => {};
  public constructor(props: ScheduleTimelineProps) {
    super(props);
    this.state = {
      perPage: undefined,
      firstDate: props.availableDates[0],
      scrollLocation: '0',
    };
    this.boundResizeHandler = this.handleResize.bind(this);
  }

  public componentDidMount() {
    this.handleResize();
    this.checkSelectedDateIsVisible();
    window.addEventListener('resize', this.boundResizeHandler);
  }

  public componentWillUnmount() {
    window.removeEventListener('resize', this.boundResizeHandler);
  }

  public render() {
    const { availableDates } = this.props;
    const { scrollLocation } = this.state;
    const { left, right } = this.checkMore();
    const leftArrow = ['schedule-timeline__arrow', left ? 'schedule-timeline__arrow--active' : ''];
    const rightArrow = ['schedule-timeline__arrow', 'schedule-timeline__arrow--right', right ? 'schedule-timeline__arrow--active' : ''];
    const scrollLeft = this.scroll.bind(this, 'LEFT');
    const scrollRight = this.scroll.bind(this, 'RIGHT');
    const contentStyles = { left: scrollLocation };
    return (
      <div className="schedule-timeline__container">
        <div onClick={scrollLeft} className={leftArrow.join(' ')}><i className="fa fa-chevron-left"/></div>
        <div className="schedule-timeline__mask">
          <div className="schedule-timeline__content" style={contentStyles}>
            {this.dateElements()}
          </div>
        </div>
        <div onClick={scrollRight} className={rightArrow.join(' ')}><i className="fa fa-chevron-right"/></div>
      </div>
    );
  }

  private scroll(direction: 'LEFT' | 'RIGHT' ) {
    let newFrontIndex;
    // checks it has more to scroll to
    const { left, right } = this.checkMore();
    const { availableDates } = this.props;
    const { perPage } = this.state;
    if (direction === 'LEFT' && left) {
      newFrontIndex = this.indexOfFirstDate() - perPage;
      return this.safelyScrollTo(newFrontIndex);
    } else if (direction === 'RIGHT' && right) {
      newFrontIndex = this.indexOfFirstDate() + perPage;
      return this.safelyScrollTo(newFrontIndex);
    }
  }

  private safelyScrollTo(frontIndex: number) {
    const { perPage } = this.state;
    const { availableDates } = this.props;
    // check that left has content
    let safeIndex = frontIndex;
    if (safeIndex < 0) {
      safeIndex = 0;
    } else if (safeIndex + perPage >= availableDates.length) {
      safeIndex = availableDates.length - perPage;
    }

    this.scrollTo(safeIndex);
  }

  private scrollTo(frontIndex: number) {
    const { perPage } = this.state;
    this.setState({
      firstDate: this.props.availableDates[frontIndex],
      scrollLocation: `-${frontIndex / perPage * 100}%`,
    });
  }

  private dateElements() {
    const { availableDates, setDate, selectedDate } = this.props;
    return availableDates.map((date: any) => {
      const selectDate = () => setDate(date);
      let classes = ['schedule-timeline__item'];
      if (date === selectedDate) {
        classes.push('schedule-timeline__item--selected');
      }
      return (
        <div className={classes.join(' ')} onClick={selectDate} key={date}>
          <div className="schedule-timeline__day">{moment(date).format('dddd')}</div>
          <div className="schedule-timeline__date">{moment(date).format('MMMM Do')}</div>
        </div>
      );
    });
  }

  private perPageForResolution() {
    const browserWidth = window.innerWidth;
    const size = {
      large: {
        width: 1301,
        perPage: 5
      },
      medium: {
        width: 769,
        perPage: 3
      }
    };
    if (browserWidth >= size.large.width) {
      return size.large.perPage;
    } else if (browserWidth >= size.medium.width) {
      return size.medium.perPage;
    }
    // for mobile
    return 1;
  }

  // currently assuming we have all data
  private checkMore() {
    const { availableDates } = this.props;
    const { perPage } = this.state;
    const firstDateIndex = this.indexOfFirstDate();

    return {
      left: firstDateIndex !== 0,
      right: firstDateIndex + perPage !== availableDates.length
    };
  }

  private indexOfFirstDate() {
    const { availableDates } = this.props;
    const { firstDate } = this.state;
    return availableDates.indexOf(firstDate);
  }

  private handleResize() {
    const amount = this.perPageForResolution();
    if (amount !== this.state.perPage) {
      this.setState({
        perPage: amount
      });
    }
  }

  private checkSelectedDateIsVisible() {
    const { selectedDate, availableDates } = this.props;
    const desiredIndex = availableDates.indexOf(selectedDate);
    setTimeout(() => this.safelyScrollTo(desiredIndex - 1), 0);
  }
}
