import * as _ from 'lodash';
import * as React from 'react';
import * as moment from 'moment';
import { Link } from 'react-router-dom';
import { bindActionCreators, Dispatch } from 'redux';
import { connect } from 'react-redux';
import {
  denormalize,
  SeriesSchema,
  ScheduledEvent,
  SeriesModel,
  getTeams,
  TournamentStageSchema,
  TournamentModel,
  getSeasons,
  SeasonSchema,
  SeasonModel,
 } from 'tourney-sdk-react';

import { fetchSchedule } from 'actions';
import { removeDisqualified, orderTournaments } from 'common/helpers';
import { AppModel } from 'state/model';
import { TickerSeries } from 'ui/components/ticker-series';
import { TOURNAMENTS, ALL_TOURNAMENTS, SEASON_NAME, TICKER_THRESHOLD } from 'common/constants';

require('./styles.scss');

export class TickerNavComponent extends React.Component<any, any> {
  public static mapStateToProps(state: AppModel, ownProps: any) {
    const orderedSeries = removeDisqualified(denormalize(state.schedule.items, [ScheduledEvent], state.tourney));
    const seasons = _.values(denormalize(_.keys(state.tourney.seasons), [SeasonSchema], state.tourney));
    const selectedSeason = seasons.find((s: SeasonModel) => s.name === SEASON_NAME) as SeasonModel;
    // loading checks that these have loaded: season, tournaments, series from schedule, teams for schedule
    const loading = !selectedSeason || !_.some(selectedSeason.tournaments) || orderedSeries.length === 0 || _.keys(state.tourney.teams).length === 0;
    return {
      selectedSeason,
      schedule: state.schedule,
      orderedSeries,
      loading
    };
  }

  public static mapDispatchToProps(dispatch: Dispatch<AppModel>, ownProps: AppModel) {
    return {
      dispatch
    };
  }

  public boundScrollHandler: (event: any) => {};
  public boundResizeHandler: (event: any) => {};
  constructor(props: any) {
    super(props);
    this.state = {
      loading: true,
      selectedEvent: undefined,
      selecting: false,
      scrolledThreshold: false,
      firstSeries: undefined,
      perPage: 3,
      interacted: false
    };
    this.boundScrollHandler = this.handleScroll.bind(this);
    this.boundResizeHandler = _.debounce(this.handleResize.bind(this), 500);
  }

  public componentDidMount() {
    // this initializes it to all

    if (!this.props.loading) {
      this.selectEvent(undefined);
    }
    this.handleResize();
    window.addEventListener('scroll', this.boundScrollHandler);
    window.addEventListener('resize', this.boundResizeHandler);
  }

  public componentWillUnmount() {
    window.removeEventListener('scroll', this.boundScrollHandler);
    window.removeEventListener('resize', this.boundResizeHandler);
  }

  public componentWillUpdate(nextProps: any) {
    if (this.props.loading !== nextProps.loading) {
      this.selectEvent(undefined);
    }

    // check live updates
    if (!this.props.loading && !this.state.interacted && this.props.orderedSeries !== nextProps.orderedSeries) {
      this.selectEvent(this.state.selectedEvent);
    }
  }

  public render() {
    const { scrolledThreshold, perPage, selectedEvent } = this.state;
    const classes = ['ticker-nav'];
    if (scrolledThreshold) {
      classes.push('scrolled');
    }

    let containerStyles = {};
    if (!this.props.loading) {
      const idx = this.indexOfFirstSeries();
      containerStyles = { left: `${-idx * (100 / perPage)}%` };
    }
    const { left, right } = this.checkMore();
    const leftClasses = ['ticker-arrow', left ? 'enabled' : 'disabled'];
    const rightClasses = ['ticker-arrow', 'right', right ? 'enabled' : 'disabled'];

    return (
      <section className={classes.join(' ')}>
        <div className="ticker-container">
          {this.renderToggle()}
          {this.renderDropdown()}
          <div className="ticker-content-mask">
            <div className="ticker-content" style={containerStyles}>
              {this.renderSeries()}
            </div>
          </div>
          <div className={leftClasses.join(' ')} onClick={this.scroll.bind(this, 'LEFT')}><i className="fa fa-chevron-left" /></div>
          <div className={rightClasses.join(' ')} onClick={this.scroll.bind(this, 'RIGHT')}><i className="fa fa-chevron-right" /></div>
        </div>
      </section>
    );
  }

  private handleResize() {
    const amount = this.perPageForResolution();
    if (amount !== this.state.perPage) {
      this.setState({
        perPage: amount
      });
    }
  }

  private perPageForResolution() {
    const browserWidth = window.innerWidth;
    const size = {
      large: {
        width: 1301,
        perPage: 3
      },
      medium: {
        width: 769,
        perPage: 2
      }
    };
    if (browserWidth >= size.large.width) {
      return size.large.perPage;
    } else if (browserWidth >= size.medium.width) {
      return size.medium.perPage;
    }
    // for mobile
    return 0;
  }

  private renderToggle() {
    const { selectedEvent, selecting } = this.state;
    const arrow = selecting
      ? 'fa-chevron-up'
      : 'fa-chevron-down';
    const arrows = <div className="ticker-option-arrows"><i className={['fa', arrow].join(' ')} /></div>;
    const toggle = this.toggleDropdown.bind(this);
    return (
      <div className="ticker-toggle" onClick={toggle}>
        {this.renderEventElement(selectedEvent)}
        {arrows}
      </div>
    );
  }

  private renderDropdown() {
    const { selectedSeason, loading } = this.props;
    const { selecting } = this.state;
    if (loading) {
      return undefined;
    }
    const allTournaments = orderTournaments(selectedSeason.tournaments);
    const rlcsTournaments = allTournaments.filter((tournament: TournamentModel) => {
      return tournament.name.match(/RLCS/);
    });
    const rlcsDropDownItem = {
      name: 'RLCS',
      logo: require('common/assets/logo_s4.svg'),
      tournaments: rlcsTournaments
    };
    const rlrsTournaments = allTournaments.filter((tournament: TournamentModel) => {
      return tournament.name.match(/RLRS/);
    });
    const rlrsDropDownItem = {
      name: 'RLRS',
      logo: require('common/assets/logo-rlrs.svg'),
      tournaments: rlrsTournaments
    };
    const items = [undefined, rlcsDropDownItem, rlrsDropDownItem].map((event: any) => {
      const selectItem = this.selectEvent.bind(this, event);
      const key = event ? event.name : 'none';
      return (
        <div key={key} className="event-option" onClick={selectItem}>
          {this.renderEventElement(event)}
        </div>
      );
    });

    const classes = ['ticker-dropdown', selecting ? '' : 'hidden'];
    return (
      <div className={classes.join(' ')}>
        {items}
      </div>
    );
  }

  private toggleDropdown() {
    this.setState({
      selecting: !this.state.selecting
    });
  }

  private selectEvent(event: any) {
    const content = this.getFilteredContent(event);
    let nextSeries = this.getFirstSeries(content);
    // check make sure that nextSeries is always "safe" (aka looks good)
    const currentIdx = content.indexOf(nextSeries);
    const safeIndex = this.safeScrollIndex(currentIdx, content);
    nextSeries = content[safeIndex];
    this.setState({
      selectedEvent: event,
      selecting: false,
      firstSeries: nextSeries,
    });
  }

  private renderEventElement(event: any) {
    if (!event) {
      event = {
        name: 'ALL',
        logo: TOURNAMENTS.RLCS.logo
      };
    }
    const name = event.name;
    return (
      <div className="event-element">
        <img className="event-image" src={event.logo} />
        <div className="event-text">{name}</div>
      </div>
    );
  }

  private getFilteredContent(selectedEvent: any) {
    const { loading } = this.props;

    let content: SeriesModel[] = this.props.orderedSeries.filter((item: any) => item.opponents);
    // if unset (ALL) it skips filter process
    if (selectedEvent) {
      const allStageIds = selectedEvent.tournaments.reduce((memo: any[], tournament: TournamentModel) => {
        return [...memo, ...tournament.stage_ids];
      }, []);
      const hashedStages = allStageIds.reduce((memo: any, id: string) => {
        return { ...memo, [id]: true };
      }, {});
      content = content.filter((series: SeriesModel) => hashedStages[series.tournament_stage_id]);
    }
    return content;
  }

  private renderSeries() {
    const { loading } = this.props;
    const { selectedEvent } = this.state;

    if (loading) {
      const loader = require('common/assets/rlcs_loader.svg') as string;
      return <img src={loader} className="ticker-loader" />;
    }

    const content = this.getFilteredContent(selectedEvent);
    let seriesList = content.map((s: SeriesModel) => <TickerSeries key={s.id} series={s}/>);
    return seriesList;
  }

  private handleScroll() {
    const pastThreshold = window.scrollY > TICKER_THRESHOLD;
    if (this.state.scrolledThreshold !== pastThreshold) {
      this.setState({
        scrolledThreshold: pastThreshold
      });
    }
  }

  private checkMore() {
    const { perPage, selectedEvent } = this.state;
    const content = this.getFilteredContent(selectedEvent);
    const firstIndex = this.indexOfFirstSeries();

    return {
      left: firstIndex !== 0,
      right: firstIndex + perPage !== content.length
    };
  }

  private scroll(direction: 'LEFT' | 'RIGHT' ) {
    const { availableDates } = this.props;
    const { perPage, selectedEvent } = this.state;
    const { left, right } = this.checkMore();
    const content = this.getFilteredContent(selectedEvent);

    // setting this will make sure that live updates won't keep bringing you back.
    this.setState({ interacted: true });
    if (direction === 'LEFT' && left) {
      const newFrontIndex = this.indexOfFirstSeries() - perPage;
      return this.safelyScrollTo(newFrontIndex);
    } else if (direction === 'RIGHT' && right) {
      const newFrontIndex = this.indexOfFirstSeries() + perPage;
      return this.safelyScrollTo(newFrontIndex);
    }
  }

  private safeScrollIndex(frontIndex: number, content: SeriesModel[]) {
    const { perPage } = this.state;
    let safeIndex = frontIndex;
    if (safeIndex < 0) {
      safeIndex = 0;
    } else if (safeIndex + perPage >= content.length) {
      safeIndex = content.length - perPage;
    }

    return safeIndex;
  }

  private safelyScrollTo(frontIndex: number) {
    const { perPage, selectedEvent } = this.state;
    const content = this.getFilteredContent(selectedEvent);
    // check that left has content
    const safeIndex = this.safeScrollIndex(frontIndex, content);
    this.scrollTo(safeIndex);
  }

  private scrollTo(frontIndex: number) {
    const { perPage, selectedEvent } = this.state;
    const content = this.getFilteredContent(selectedEvent);
    this.setState({
      firstSeries: content[frontIndex],
      scrollLocation: `-${frontIndex / perPage * 100}%`,
    });
  }

  private indexOfFirstSeries() {
    const { firstSeries, selectedEvent } = this.state;
    if (!firstSeries) {
      return -1;
    }
    return _.findIndex(this.getFilteredContent(selectedEvent), (series: SeriesModel) => {
      return series.id === firstSeries.id;
    });
  }

  private getFirstSeries(seriesList: SeriesModel[]) {
    let nextSeries = seriesList[0];
    for (let series of seriesList) {
      const scoreArr = series.opponents.map((opp: any) => opp.score);
      // is live
      if (series.winner_id === null && _.some(scoreArr)) {
        return series;
      }

      if (series.winner_id) {
        nextSeries = series;
      }
    }
    return nextSeries;
  }
}

export const TickerNav = connect(TickerNavComponent.mapStateToProps, TickerNavComponent.mapDispatchToProps)(TickerNavComponent);
