import * as _ from 'lodash';
import * as React from 'react';
import { connect } from 'react-redux';
import { bindActionCreators, Dispatch } from 'redux';
import { denormalize, ArticleSchema, ArticleModel } from 'tourney-sdk-react';

import { fetchLive, fetchStreams, fetchArticles, fetchVideos } from 'actions';
import { AppModel } from 'state/model';
import { HeroController } from 'ui/components/hero-controller';
import { LiveTakeover } from 'ui/components/live-takeover';
import { SponsorBar } from 'ui/components/sponsor-bar';
import { ArticleSection } from 'ui/components/article-section';
import { StreamSection } from 'ui/components/stream-section';

const NUM_HERO_SLIDES = 3;
const NUM_MORE_NEWS = 100;

export class NewsPageComponent extends React.Component<any, any> {
  public static mapStateToProps(state: AppModel) {
    const articles = denormalize(state.article.articles, [ArticleSchema], state.tourney);
    return {
      stream: state.stream,
      articles
    };
  }

  public static mapDispatchToProps(dispatch: Dispatch<AppModel>, ownProps: AppModel) {
    return bindActionCreators({
      fetchStreams,
      fetchArticles,
      fetchVideos,
    }, dispatch);
  }

  public constructor(...args: any[]) {
    super(args[0]);
  }

  public componentDidMount() {
    this.props.fetchStreams();
    this.props.fetchArticles();
    this.props.fetchVideos();
  }

  public render() {
    const { stream, articles } = this.props;
    const classes = ['news'];

    return (
      <div className={classes.join(' ')}>
        {this.renderHeroContent()}
        <SponsorBar />
        <div className="news-listings">
          {this.renderArticleSection()}
          {this.renderStreamsSection()}
        </div>
      </div>
    );
  }

  private renderHeroContent() {
    const { stream, articles } = this.props;

    // default to loading if content is empty (meaning it hasn't loaded just yet)
    let mainContent: JSX.Element = (
      <div className="hero-slider">
        <div className="hero-slider__loading"/>
      </div>
    );
    if (stream.live) {
      mainContent = <LiveTakeover />;
    } else if (articles.length > 0) {
      const heroArticles = articles.slice(0, NUM_HERO_SLIDES);
      mainContent = <HeroController articles={heroArticles} />;
    }
    return mainContent;
  }

  private renderArticleSection() {
    const { articles } = this.props;
    const articleList = articles;
    // const articleList = articles.slice(NUM_HERO_SLIDES, NUM_MORE_NEWS + NUM_HERO_SLIDES);
    // const articleList: ArticleModel[] = []; //testing visuals with no articles
    return articleList.length > 0
      ? <ArticleSection articles={articleList}/>
      : undefined;
  }

  private renderStreamsSection() {
    const { stream } = this.props;

    return <StreamSection stream={stream}/>;
  }
}

export const NewsPage = connect(NewsPageComponent.mapStateToProps, NewsPageComponent.mapDispatchToProps)(NewsPageComponent);
