import { keys, keyBy, values, find, some, sortBy } from 'lodash';
import * as React from 'react';
import { connect } from 'react-redux';
import { Link } from 'react-router-dom';
import { bindActionCreators, Dispatch } from 'redux';
import {
  denormalize,
  TournamentStageSchema,
  TournamentStageModel,
  SeasonSchema,
  SeasonModel,
  TeamSchema,
  TeamModel,
  TournamentModel,
} from 'tourney-sdk-react';

import { SEASON_NAME, STAGE_TYPES, SINGLE_ELIM_FORMAT, DOUBLE_ELIM_FORMAT, FINALS_FORMAT, BracketType } from 'common/constants';
import { orderTournaments, getPrettyNames } from 'common/helpers';
import { AppModel } from 'state/model';
import { Standings } from 'ui/components/standings';
import { Brackets } from 'ui/components/brackets';

import './styles.scss';

interface ResultsListPageProps {
  loading: boolean;
  selectedSeason: SeasonModel;
  teamsById: {[key: string]: TeamModel};
}

const PROMOTION_TOURNAMENTS = [
  'NA RLCS/RLRS Promotion',
  'EU RLCS/RLRS Promotion',
];

export class ResultsListPageComponent extends React.Component<any, any> {
  public static mapStateToProps(state: AppModel, ownProps: any): ResultsListPageProps {
    const seasons = values(denormalize(keys(state.tourney.seasons), [SeasonSchema], state.tourney));
    const selectedSeason = seasons.find((s: SeasonModel) => s.name === SEASON_NAME) as SeasonModel;
    const teams = denormalize(keys(state.tourney.teams), [TeamSchema], state.tourney);
    const loading = !selectedSeason || !some(selectedSeason.tournaments) || teams.length === 0;
    return {
      loading,
      selectedSeason,
      teamsById: keyBy(teams, 'id')
    };
  }

  public static mapDispatchToProps(dispatch: Dispatch<AppModel>, ownProps: AppModel) {
    return {
      dispatch
    };
  }

  public render() {
    const { loading, selectedSeason } = this.props;

    let stages;
    let brackets;
    let promoBrackets;
    let finalBracket;
    if (!loading) {
      const sortedTournaments = orderTournaments(selectedSeason.tournaments);
      const roundRobinStages = sortedTournaments.reduce((memo: TournamentStageModel[], tournament: TournamentModel) => {
        return [...memo, ...tournament.stages.filter((stage: TournamentStageModel) => stage.type === STAGE_TYPES.ROUND_ROBIN)];
      }, []);
      const singleElim = sortedTournaments.reduce((memo: TournamentStageModel[], tournament: TournamentModel) => {
        return [...memo, ...tournament.stages.filter((stage: TournamentStageModel) => stage.type === STAGE_TYPES.SINGLE_ELIMINATION)];
      }, []);
      const doubleElims = sortedTournaments.reduce((memo: TournamentStageModel[], tournament: TournamentModel) => {
        return [...memo, ...tournament.stages.filter((stage: TournamentStageModel) => {
          return stage.type === STAGE_TYPES.DOUBLE_ELIMINATION
            && PROMOTION_TOURNAMENTS.indexOf(stage.tournament.name) !== -1;
        })];
      }, []).reverse();
      const finals = sortedTournaments.reduce((memo: TournamentStageModel[], tournament: TournamentModel) => {
        return [...memo, ...tournament.stages.filter((stage: TournamentStageModel) => {
          return stage.type === STAGE_TYPES.DOUBLE_ELIMINATION
            && stage.tournament.name === 'RLCS S4 World Championship';
        })];
      }, []);
      promoBrackets = doubleElims.map((stage) => this.renderTournamentBracket(stage, DOUBLE_ELIM_FORMAT));
      brackets = singleElim.map((stage) => this.renderTournamentBracket(stage, SINGLE_ELIM_FORMAT));
      finalBracket = finals.map((stage) => this.renderTournamentBracket(stage, FINALS_FORMAT));
      stages = roundRobinStages.map(this.renderTournamentStandings.bind(this));
    }
    return (
      <div className="results-list__background">
        <div className="results-list__container page-content">
          <h1>Results</h1>
          {finalBracket}
          {promoBrackets}
          {brackets}
          <ul className="results-list__contents">
            {stages}
          </ul>
        </div>
      </div>
    );
  }

  private renderTournamentBracket(tournamentStage: TournamentStageModel, format: BracketType[]) {
    return (
      <Brackets tournamentStage={tournamentStage} format={format} key={tournamentStage.id} />
    );
  }

  private renderTournamentStandings(tournamentStage: TournamentStageModel) {
    const { teamsById } = this.props;
    const bgStyle = { backgroundImage: `url(${tournamentStage.tournament.logo.original})` };
    return (
      <div key={tournamentStage.id} className="standing-preview__container">
        <div className="standing-preview__heading">
          <div className="standing-preview__heading-logo" style={bgStyle} />
          <div className="standing-preview__heading-tournament">{tournamentStage.tournament.name}</div>
          <div className="standing-preview__heading-stage">{tournamentStage.name}</div>
        </div>
        <Standings tournamentStage={tournamentStage} teamsById={teamsById} />
      </div>
    );
  }
}

export const ResultsListPage = connect(ResultsListPageComponent.mapStateToProps, ResultsListPageComponent.mapDispatchToProps)(ResultsListPageComponent);
