import * as _ from 'lodash';
import * as React from 'react';
// import * as moment from 'moment';
import * as moment from 'moment-timezone';

import { connect } from 'react-redux';
import { Link } from 'react-router-dom';
import { bindActionCreators, Dispatch } from 'redux';
import {
  denormalize,
  SeriesSchema,
  ScheduledEvent,
  SeriesModel,
  TournamentStageSchema,
  TournamentStageModel,
  getTournaments,
  TournamentSchema,
  TournamentModel,
  getTeams,
  EventModel
 } from 'tourney-sdk-react';

import { fetchSchedule } from 'actions';
import { removeDisqualified } from 'common/helpers';
import { AppModel } from 'state/model';
import { ScheduleTimeline } from 'ui/components/schedule-timeline';
import { ScheduleEvents } from 'ui/components/schedule-events';

require('./styles.scss');

const DATE_FORMAT = 'YYYY-MM-DD';

function groupSeriesByDay(orderedSeries: (SeriesModel | EventModel)[]) {
  const seriesByDate = _.groupBy(orderedSeries, (s: any) => {
    // events use starts at instead of scheduled_at
    const start = s.scheduled_at ? s.scheduled_at : s.starts_at;
    return moment(start).tz('America/Los_Angeles').format(DATE_FORMAT);
  });
  const orderedDates = _.chain(seriesByDate)
    .keys()
    .sortBy((d: string) => moment(d))
    .value();
  return { seriesByDate, orderedDates };
}

export class SchedulePageComponent extends React.Component<any, any> {
  public static mapStateToProps(state: AppModel, ownProps: any) {
    const orderedSeries = removeDisqualified(denormalize(state.schedule.items, [ScheduledEvent], state.tourney));
    const { seriesByDate, orderedDates } = groupSeriesByDay(orderedSeries);
    // this might not be the best way, but we need some mapping right now
    const tournamentsById = _.keyBy(denormalize(state.tourney.tournaments, [TournamentSchema], state.tourney), 'id');
    const tournamentStagesIds = _.keys(state.tourney.tournament_stages);
    const tournamentStages = denormalize(tournamentStagesIds, [TournamentStageSchema], state.tourney);
    const loading = state.schedule.items.length === 0 || tournamentStagesIds.length === 0 || _.keys(state.tourney.teams).length === 0;
    return {
      schedule: state.schedule,
      orderedSeries,
      seriesByDate,
      orderedDates,
      tournamentStages,
      tournamentsById,
      loading
    };
  }

  public static mapDispatchToProps(dispatch: Dispatch<AppModel>, ownProps: AppModel) {
    return {
      dispatch
    };
  }

  constructor(props: any) {
    super(props);
    this.state = {
      // loading: true,
      selectedDate: undefined,
      availableDates: [],
    };
  }

  public componentDidMount() {
    if (!this.props.loading) {
      this.setState({
        selectedDate: this.selectBestInitialDate()
      });
    }
  }

  public componentWillUpdate(nextProps: any) {
    if (this.props.loading !== nextProps.loading) {
      if (this.props.orderedDates[0]) {
        this.setState({
          selectedDate: this.selectBestInitialDate()
        });
      }
    }
  }

  public render() {
    const { selectedDate } = this.state;
    const { seriesByDate, orderedDates, tournamentStages, loading, tournamentsById } = this.props;

    let timelineContent = <div />;
    let eventContent = <div />;
    let loadingElement = <div />;
    if (!loading && selectedDate) {
      timelineContent = (
        <ScheduleTimeline
          selectedDate={selectedDate}
          setDate={this.setDate.bind(this)}
          availableDates={orderedDates}
        />
      );
      eventContent = (
        <ScheduleEvents
          availableDates={orderedDates}
          selectedDate={selectedDate}
          seriesByDate={seriesByDate}
          tournamentStages={tournamentStages}
          tournamentsById={tournamentsById}
        />
      );
    }else {
      const loaderImg = require('common/assets/rlcs_loader.svg') as string;
      loadingElement = <img className="schedule__loader" src={loaderImg} />;
    }

    return (
      <section className="schedule__page">
        <div className="schedule__container">
          <h1 className="schedule__heading">Schedule</h1>
          {timelineContent}
          <div className="schedule__content">
            {eventContent}
            {loadingElement}
          </div>
        </div>
      </section>
    );
  }

  private setDate(date: any) {
    this.setState({
      selectedDate: date,
    });
  }

  private findAllDates(events: any[]) {
    return _.chain(events)
      .map((e: any) => e.series)
      .flatten()
      .groupBy((s: any) => s.scheduled_at.format(DATE_FORMAT))
      .keys()
      .sortBy((d) => moment(d))
      .value();
  }

  private selectBestInitialDate() {
    const { orderedSeries } = this.props;
    let nextSeries = orderedSeries[0];
    for (let series of orderedSeries) {
      if (series.starts_at && moment(series.starts_at) > moment()) {
        nextSeries = series;
        break;
      } else if (series.scheduled_at && !series.winner_id) {
        nextSeries = series;
        break;
      }
    }
    const nextDate = nextSeries.starts_at ? nextSeries.starts_at : nextSeries.scheduled_at;
    return moment(nextDate).tz('America/Los_Angeles').format(DATE_FORMAT);
  }
}

export const SchedulePage = connect(SchedulePageComponent.mapStateToProps, SchedulePageComponent.mapDispatchToProps)(SchedulePageComponent);
