import { stateToHTML } from 'draft-js-export-html';
import { convertFromRaw } from 'draft-js';
import { keys, keyBy } from 'lodash';
import * as React from 'react';
import { connect } from 'react-redux';
import { Link } from 'react-router-dom';
import { bindActionCreators, Dispatch } from 'redux';
import {
  denormalize,
  TeamSchema,
  TeamModel,
  getTeam,
  UserModel,
  TournamentSchema,
} from 'tourney-sdk-react';

import { AppModel } from 'state/model';
import { teamLogoWithFallback } from 'common/helpers';
import { Player } from 'ui/components/player';
import { TeamNav } from 'ui/components/team-nav';
import { TeamSchedule } from 'ui/components/team-schedule';
import { TeamStanding } from 'ui/components/team-standing';
import { TeamSocialLinks } from 'ui/components/team-socials';

require('./styles.scss');

export class TeamDetailPageComponent extends React.Component<any, any> {
  public static mapStateToProps(state: AppModel, ownProps: any) {
    const teamsBySlug = keyBy(denormalize(keys(state.tourney.teams), [TeamSchema], state.tourney), 'slug');
    const team = teamsBySlug[ownProps.match.params.id];
    const tournaments = denormalize(keys(state.tourney.tournaments), [TournamentSchema], state.tourney);
    const loading = !team || tournaments.length === 0;
    return {
      team,
      tournaments,
      loading
    };
  }

  public static mapDispatchToProps(dispatch: Dispatch<AppModel>, ownProps: AppModel) {
    return bindActionCreators({
      getTeam
    }, dispatch);
  }

  public render() {
    const { team, tournaments, loading } = this.props;

    let content: JSX.Element;
    if (loading) {
      content = <div className="page-loading extra-margin" />;
    } else {
      content = (
        <div>
          <TeamNav team={team} tournaments={tournaments}/>
          {this.renderCover()}
          {this.renderRoster()}
          {this.renderStandings()}
        </div>
      );
    }

    return (
      <div className="teams-detail">
        {content}
      </div>
    );
  }

  private renderCover() {
    const { team, tournaments } = this.props;
    const img = require('common/assets/cover_generic1.jpg') as string;
    const backgroundStyle = { backgroundImage: `url(${img})` };
    const logo = teamLogoWithFallback(team);
    const backgroundLogo = { backgroundImage: `url(${logo})` };
    const fallbackCover = require('common/assets/team_fallback.png') as string;
    const cover = team.cover && team.cover.original
      ? team.cover.original
      : fallbackCover;
    const bgStyles = { backgroundImage: `url(${cover})` };
    const descriptionElement = this.getDescription();

    return (
      <section className="team-cover" id="about" style={backgroundStyle}>
        <div className="team-cover__player-background" style={bgStyles}/>
        <div className="team-cover__container section-container">
          <div className="team-cover__team-info">
            <div className="team-cover__team">
              <div className="team-cover__logo" style={backgroundLogo} />
              <div className="team-cover__name">{team.name}</div>
            </div>
            <TeamStanding team={team} tournaments={tournaments} />
            <TeamSocialLinks team={team} />
          </div>
        </div>
        <div className="team-cover__description">
          {descriptionElement}
        </div>
      </section>
    );
  }

  private getDescription() {
    const { team } = this.props;
    let htmlContent;
    try {
      const content = convertFromRaw(JSON.parse(team.description));
      let options = {
        blockRenderers: {
          'atomic:image': (block: any) => {
            let data = block.getData();
            if (data.get('src')) {
              return `<img src="${data.get('src')}" />`;
            }
          },
        },
      };
      htmlContent = stateToHTML(content, options);
    } catch (error) {
      htmlContent = '';
    }

    return htmlContent !== ''
      ? <div className="team-cover__description-content section-container" dangerouslySetInnerHTML={{ __html: htmlContent }} />
      : undefined;
  }

  private renderStandings() {
    const { team } = this.props;
    return (
    <section className="team-schedule" id="matches">
        <div className="team-schedule__container section-container">
          <h2>Matches</h2>
          <TeamSchedule team={team}/>
        </div>
      </section>
    );
  }

  private renderRoster() {
    const { team } = this.props;
    const playerElements = team.users.map((player: UserModel) => <Player player={player} />);
    return (
      <section className="team-roster" id="roster">
        <div className="team-roster__container section-container">
          <h2>Roster</h2>
          <div className="team-roster__players">
            {playerElements}
          </div>
        </div>
      </section>
    );
  }
}

export const TeamDetailPage = connect(TeamDetailPageComponent.mapStateToProps, TeamDetailPageComponent.mapDispatchToProps)(TeamDetailPageComponent);
