import { keys, values } from 'lodash';
import * as React from 'react';
import { connect } from 'react-redux';
import { Link } from 'react-router-dom';
import { bindActionCreators, Dispatch } from 'redux';

import { AppModel } from 'state/model';
import {
  denormalize,
  TeamSchema,
  TeamModel,
  getTeams,
  TournamentSchema,
  TournamentModel,
  SeasonSchema,
  SeasonModel,
} from 'tourney-sdk-react';

import { SEASON_NAME, TOURNAMENT_CLASSES, TOURNAMENT_WHITELIST } from 'common/constants';
import { teamLogoWithFallback, orderTournaments } from 'common/helpers';

import './styles.scss';

function getTournamentByTeamId(tournaments: TournamentModel[]) {
  return tournaments.reduce((memo: any, tournament: TournamentModel) => {
    const teamObj = tournament.teams.reduce((teamMemo: any, team: TeamModel) => {
      return {
        ...teamMemo,
        [team.id]: tournament
      };
    }, {});
    return {
      ...memo,
      ...teamObj,
    };
  }, {});
}

export class TeamListPageComponent extends React.Component<any, any> {
  public static mapStateToProps(state: AppModel, ownProps: any) {
    const teams = denormalize(keys(state.tourney.teams), [TeamSchema], state.tourney);
    const seasons = values(denormalize(keys(state.tourney.seasons), [SeasonSchema], state.tourney));
    const season = seasons.find((s: SeasonModel) => s.name === SEASON_NAME) as SeasonModel;
    const loading = !season || teams.length === 0;
    const tournamentsByTeamId = loading ? undefined : getTournamentByTeamId(season.tournaments);
    return {
      teams,
      season,
      tournamentsByTeamId,
      loading
    };
  }

  public static mapDispatchToProps(dispatch: Dispatch<AppModel>, ownProps: AppModel) {
    return bindActionCreators({
      getTeams
    }, dispatch);
  }

  public constructor(props: any) {
    super(props);
    this.state = {
      teamSearch: '',
      tournamentFilter: undefined,
      dropdownOpen: false,
      filteredTeams: []
    };
  }

  public render() {
    const { season, loading } = this.props;
    const img = require('common/assets/cover_generic2.jpg') as string;
    const backgroundStyle = { backgroundImage: `url(${img})` };

    let content: JSX.Element;
    if (loading) {
      content = <div className="page-loading" />;
    } else {
      // const teamElements = this.getFilteredTeams().map(this.renderTeam.bind(this));
      const tournamentElements = orderTournaments(season.tournaments)
        .filter((tournament: TournamentModel) => TOURNAMENT_WHITELIST[tournament.name])
        .map(this.renderTournamentTeams.bind(this));
      content = (
        <div>
          {/* {this.renderFilter()} */}
          <ul className="teams-page__list">
            {tournamentElements}
          </ul>
        </div>
      );
    }

    return (
      <div className="teams-page" style={backgroundStyle}>
        <div className="teams-page__content">
          <h1>Teams</h1>
          {content}
        </div>
      </div>
    );
  }

  private renderTournamentTeams(tournament: TournamentModel) {
    const img = require('common/assets/logo_s4.svg') as string;
    const teamElements = tournament.teams.map((team: TeamModel) => this.renderTeam(team, tournament));
    return (
      <li className="teams-page__section" key={tournament.id}>
        <div className="teams-page__tournament-heading">
          <img src={tournament.logo.original} className="teams-page__tournament-logo"/>
          {tournament.name}
        </div>
        <div className="teams-page__team-list">
          {teamElements}
        </div>
      </li>
    );
  }

  private getFilteredTeams(): TeamModel[] {
    const { season } = this.props;
    const { teamSearch, tournamentFilter } = this.state;
    let filteredTeams: TeamModel[];

    if (tournamentFilter) {
      filteredTeams = [...tournamentFilter.teams];
    } else {
      filteredTeams = orderTournaments(season.tournaments)
      .reduce((memo: TeamModel[], tournament: any) => {
        return [...memo, ...tournament.teams];
      }, []);
    }

    if (teamSearch !== '') {
      filteredTeams = filteredTeams.filter((team: TeamModel) => {
        return team.name.toLowerCase().indexOf(teamSearch.trim().toLowerCase()) !== -1;
      });
    }

    return filteredTeams;
  }

  private handleTextChange(event: any) {
    this.setState({ teamSearch: event.target.value });
  }

  private clearSearch() {
    this.setState({ teamSearch: '' });
  }

  private clearTournaments(event: any) {
    event.stopPropagation();
    this.setState({
      tournamentFilter: undefined,
      dropdownOpen: false,
    });
  }

  private toggleDropdown() {
    this.setState({ dropdownOpen: !this.state.dropdownOpen });
  }

  private renderFilter() {
    const { dropdownOpen, tournamentFilter, teamSearch } = this.state;
    const { season } = this.props;

    const searchClose = teamSearch === ''
      ? undefined
      : <div className="team-filter__close" onClick={this.clearSearch.bind(this)}>x</div>;

    const tournamentClose = tournamentFilter
      ? <div className="team-filter__close" onClick={this.clearTournaments.bind(this)}>x</div>
      : <div className="team-filter__arrow"><i className="fa fa-chevron-down" /></div>;

    const dropdownElement = dropdownOpen
      ? this.renderFilterDropdown(season.tournaments)
      : undefined;
    const activeTournament = tournamentFilter
      ? tournamentFilter.name
      : <span>Tournament...</span>;
    return (
      <div className="team-filter__container">
        <div className="team-filter__search">
          <label>Filter By Team Name</label>
          <input placeholder="Team Name..." type="text" value={teamSearch} onChange={this.handleTextChange.bind(this)}/>
          {searchClose}
        </div>
        <div className="team-filter__tournament" onClick={this.toggleDropdown.bind(this)}>
          <label>Tournament</label>
          <div className="team-filter__active">
            {activeTournament}
          </div>
          {tournamentClose}
        </div>
        {dropdownElement}
      </div>
    );
  }

  private renderFilterDropdown(tournaments: TournamentModel[]) {
    const tournamentElements = orderTournaments(tournaments).map((tournament: TournamentModel) => {
      const selectTournament = () => {
        this.setState({
          tournamentFilter: tournament,
          dropdownOpen: false,
        });
      };
      return (
        <div
          className="team-filter__dropdown-item"
          key={tournament.id}
          onClick={selectTournament}
        >
          {tournament.name}
        </div>
      );
    });

    return (
      <div className="team-filter__dropdown">
        {tournamentElements}
      </div>
    );
  }

  private renderTeam(team: TeamModel, tournament: TournamentModel) {
    const { tournamentsByTeamId } = this.props;
    const link = `/teams/${team.slug}`;
    const teamLogo = teamLogoWithFallback(team);
    const tournamentName = tournamentsByTeamId[team.id].name;
    // const tournamentColorStyle = { backgroundColor:  };
    return (
      <div className={['teams-page__item', TOURNAMENT_CLASSES[tournament.name]].join(' ')} key={team.id}>
        <div className={['teams-page__second-background', TOURNAMENT_CLASSES[tournamentName] + '-bg'].join(' ')}/>
        <Link className="teams-page__link" to={link}>
          <img className="teams-page__logo" src={teamLogo} />
          <div className="teams-page__team-name">{team.name}</div>
          <div className="teams-page__hover-container">
            <div className="teams-page__hover-text">
              View Profile <i className="fa fa-arrow-right" />
            </div>
          </div>
        </Link>
      </div>
    );
  }
}

export const TeamListPage = connect(TeamListPageComponent.mapStateToProps, TeamListPageComponent.mapDispatchToProps)(TeamListPageComponent);
