import { keys, keyBy, find, reverse } from 'lodash';
import * as React from 'react';
import { connect } from 'react-redux';
import { Link } from 'react-router-dom';
import { bindActionCreators, Dispatch } from 'redux';
import {
  denormalize,
  TournamentSchema,
  TournamentModel,
  TournamentStageModel,
  TeamSchema,
  TeamModel
} from 'tourney-sdk-react';

import { STAGE_TYPES, SINGLE_ELIM_FORMAT, DOUBLE_ELIM_FORMAT } from 'common/constants';
import { getPrettyNames } from 'common/helpers';
import { AppModel } from 'state/model';
import { Brackets } from 'ui/components/brackets';
import { Standings } from 'ui/components/standings';

import './styles.scss';

interface TournamentResultsPageProps {
  loading: boolean;
  tournament: TournamentModel;
  teamsById: {[key: string]: TeamModel};
}

export class TournamentResultsPageComponent extends React.Component<any, any> {
  public static mapStateToProps(state: AppModel, ownProps: any): TournamentResultsPageProps {
    const tournaments = denormalize(keys(state.tourney.tournaments), [TournamentSchema], state.tourney);
    const tournament = tournaments.find((t: TournamentModel) => t.slug === ownProps.match.params.tournamentSlug);
    const teams = denormalize(keys(state.tourney.teams), [TeamSchema], state.tourney);
    const loading = !tournament || teams.length === 0;
    return {
      loading,
      tournament,
      teamsById: keyBy(teams, 'id')
    };
  }

  public static mapDispatchToProps(dispatch: Dispatch<AppModel>, ownProps: AppModel) {
    return {
      dispatch
    };
  }

  public render() {
    const { loading, tournament, teamsById } = this.props;

    if (loading) {
      return (
        <div className="standings-page page-content">
          <div className="standings-page__contents">
            Loading
          </div>
        </div>
      );
    }

    let stageElements = reverse(tournament.stages).map((stage: TournamentStageModel) => {
      if (stage.type === STAGE_TYPES.ROUND_ROBIN) {
        return <Standings tournamentStage={stage} teamsById={teamsById} />;
      } else if (stage.type === STAGE_TYPES.SINGLE_ELIMINATION) {
        return <Brackets tournamentStage={stage} format={SINGLE_ELIM_FORMAT} />;
      } else if (stage.type === STAGE_TYPES.DOUBLE_ELIMINATION) {
        return <Brackets tournamentStage={stage} format={DOUBLE_ELIM_FORMAT} />;
      }
    });
    let prettyName = getPrettyNames(tournament.name);
    const logo = tournament.logo.original
      ? tournament.logo.original
      : require('common/assets/logo_s4.svg');
    let bgStyle = { backgroundImage: `url(${logo})` };
    return (
      <div className="results-detail page-content">
        <div className="results-detail__contents">
          <div className="results-detail__heading">
            <div className="results-detail__heading-logo" style={bgStyle} />
            <div className="results-detail__heading-tournament">
              <span className="results-detail__name">{prettyName.name}</span><br/>
              <span className="results-detail__region">{prettyName.region}</span>
            </div>
          </div>
          {stageElements}
        </div>
      </div>
    );
  }
}

export const TournamentResultsPage = connect(TournamentResultsPageComponent.mapStateToProps, TournamentResultsPageComponent.mapDispatchToProps)(TournamentResultsPageComponent);
