import { keys, keyBy, values, find, some, sortBy } from 'lodash';
import * as React from 'react';
import { connect } from 'react-redux';
import { Link } from 'react-router-dom';
import { bindActionCreators, Dispatch } from 'redux';
import {
  denormalize,
  TournamentStageSchema,
  TournamentStageModel,
  SeasonSchema,
  SeasonModel,
  TeamSchema,
  TeamModel,
  TournamentModel,
} from 'tourney-sdk-react';

import { SEASON_NAME } from 'common/constants';
import { orderTournaments, getPrettyNames } from 'common/helpers';
import { AppModel } from 'state/model';
import { Standings } from 'ui/components/standings';

import './styles.scss';

interface TournamentResultsListPageProps {
  loading: boolean;
  selectedSeason: SeasonModel;
  teamsById: {[key: string]: TeamModel};
}

interface TreeNode {
  item: any;
  children: any[];
  depth: number;
  x: number;
  y: number;
}

export class TournamentResultsListPageComponent extends React.Component<any, any> {
  public static mapStateToProps(state: AppModel, ownProps: any): TournamentResultsListPageProps {
    const seasons = values(denormalize(keys(state.tourney.seasons), [SeasonSchema], state.tourney));
    const selectedSeason = seasons.find((s: SeasonModel) => s.name === SEASON_NAME) as SeasonModel;
    const teams = denormalize(keys(state.tourney.teams), [TeamSchema], state.tourney);
    const loading = !selectedSeason || !some(selectedSeason.tournaments) || teams.length === 0;
    return {
      loading,
      selectedSeason,
      teamsById: keyBy(teams, 'id')
    };
  }

  public static mapDispatchToProps(dispatch: Dispatch<AppModel>, ownProps: AppModel) {
    return {
      dispatch
    };
  }
  public nodes: TreeNode[];
  public constructor(props: TournamentResultsListPageProps) {
    super(props);
    this.nodes = [];
  }

  public render() {
    const { loading, selectedSeason } = this.props;

    let tournamentElements;
    if (!loading) {
      const sortedTournaments = orderTournaments(selectedSeason.tournaments);
      tournamentElements = sortedTournaments.map(this.renderTournamentListing.bind(this));
    }
    return (
      <div className="results-list__background">
        <div className="results-list__container page-content">
          <h1>Results</h1>
          <ul className="results-list__contents">
            {tournamentElements}
          </ul>
        </div>
      </div>
    );
  }

  private renderTournamentListing(tournament: TournamentModel) {
    // const bgStyle = { backgroundImage: `url(${tournament.logo.original})` };
    const prettyNames = getPrettyNames(tournament.name);
    const logo = tournament.logo.original
      ? tournament.logo.original
      : require('common/assets/logo_s4.svg');
    return (
      <div className="results-list__item-container">
        <Link to={`/tournaments/${tournament.slug}`}>
          <div className="results-list__item">
            <img className="results-list__logo" src={logo} />
            <div className="results-list__title">
                {prettyNames.name}
            </div>
            <div className="results-list__region">
                {prettyNames.region}
            </div>
          </div>
        </Link>
      </div>
    );
  }

  // private renderTournamentStandings(tournamentStage: TournamentStageModel) {
  //   const { teamsById } = this.props;
  //   const bgStyle = { backgroundImage: `url(${tournamentStage.tournament.logo.original})` };
  //   return (
  //     <div key={tournamentStage.id} className="standing-preview__container">
  //       <div className="standing-preview__heading">
  //         <div className="standing-preview__heading-logo" style={bgStyle} />
  //         <div className="standing-preview__heading-tournament">{tournamentStage.tournament.name}</div>
  //         <div className="standing-preview__heading-stage">{tournamentStage.name}</div>
  //       </div>
  //       <Standings tournamentStage={tournamentStage} teamsById={teamsById} />
  //     </div>
  //   );
  // }

  // private renderSingleEliminationBracket() {
  //   const bracket = SINGLE_ELIM_FORMAT.BRACKET;
  //   const brackets = bracket.rounds.map(this.renderRound.bind(this));
  //   return (
  //     <div className="bracket-series">
  //       {brackets}
  //     </div>
  //   );
  // }

  // private renderRound(round: any) {
  //   const series = round.series.map(this.renderSeriesItem.bind(this));
  //   return (
  //     <div className="bracket-round">
  //       <h2>{round.title}</h2>
  //       <ul className="bracket-round__list">
  //         {series}
  //       </ul>
  //     </div>
  //   );
  // }

  // private renderSeriesItem(seriesContainer: any) {
  //   const series = SERIES[seriesContainer.ident];
  //   const opponents = series.opponents.map((opp: any) => {
  //       return (
  //         <div className="series-bracket__opponent">
  //           {opp}
  //         </div>
  //       );
  //   });

  //   const classes = ['bracket-round__series'];
  //   if (seriesContainer.type) {
  //     classes.push(`bracket-round__series--${seriesContainer.type}`);
  //   }

  //   return (
  //     <li className={classes.join(' ')}>
  //       {opponents}
  //     </li>
  //   );
  // }

  // private renderBracket() {
  //   this.nodes = [];
  //   const root = this.buildTree();
  //   console.log('tree', root);
  //   console.log('nodes', this.nodes);
  //   return this.renderTree(root);
  // }

  // private buildTree() {
  //   // find root
  //   const root: TreeNode = {
  //     item: this.findRoot(),
  //     children: [],
  //     depth: 0,
  //     x: 0,
  //     y: 0,
  //   };
  //   this.buildNode(root);
  //   return root;
  // }

  // private buildNode(node: TreeNode) {
  //   const seriesList = values(SERIES);
  //   const children = seriesList.filter((series: any) => node.item === series.winner_to);
  //   const depthMap = {
  //     1: 0,
  //     2: 400,
  //     3: 200,
  //     4: 100,
  //     5: 50,
  //   };
  //   node.children = children.map((item: any, idx) => {
  //     const depth = node.depth + 1;
  //     const newY = idx ? node.y + depthMap[depth] : node.y - depthMap[depth];
  //     const childNode: TreeNode = {
  //       item,
  //       children: [],
  //       depth,
  //       x: node.x + 200,
  //       y: newY
  //     };
  //     this.nodes.push(childNode);
  //     return childNode;
  //   });
  //   if (node.children.length >= 0) {
  //     node.children.forEach((childNode: TreeNode, idx: number) => {
  //       this.buildNode(childNode);
  //     });
  //   }
  //   return node;
  // }

  // private findRoot() {
  //   const seriesList = values(SERIES);
  //   return find(seriesList, (series: any) => series.winner_to === null);
  // }

  // private renderTree(node: TreeNode) {
  //   const rootSeries = this.nodes.map(this.renderSeries.bind(this));
  //   return (
  //     <div className="bracket">
  //       {rootSeries}
  //     </div>
  //   );
  // }

  // // private renderRound() {
  // //   const roundData = [];
  // //   return (
  // //     <div className="series-round">

  // //     </div>
  // //   );
  // // }

  // private renderSeries(node: TreeNode) {
  //   const style = {
  //     top: node.y,
  //     right: node.x,
  //   };
  //   return (
  //     <div className="series-bracket" style={style}>
  //       <div className="series-bracket__opponent">
  //         {node.depth} - {node.item.opponents[0]}
  //       </div>
  //       <div className="series-bracket__opponent">
  //         {node.depth} - {node.item.opponents[1]}
  //       </div>
  //     </div>
  //   );
  // }

  // private renderStageStandings(tournamentStage: TournamentStageModel) {
  //   const stageLogo: string = tournamentStage.name.match(/RLRS/)
  //     ? require('common/assets/logo-rlrs.svg')
  //     : require('common/assets/logo_s4.svg');
  //   const link = `/results/${tournamentStage.id}`;
  //   return (
  //     <li className="results-list__item-container">
  //       <div className="results-list__item">
  //         <Link className="results-list__link" to={link}>
  //           <img className="results-list__logo" src={stageLogo} />
  //           <div className="results-list__title">{tournamentStage.name}</div>
  //         </Link>
  //       </div>
  //     </li>
  //   );
  // }
}

export const TournamentResultsListPage = connect(TournamentResultsListPageComponent.mapStateToProps, TournamentResultsListPageComponent.mapDispatchToProps)(TournamentResultsListPageComponent);
