provider "aws" {
    region = "us-west-2"
    profile = "twitch-esports-aws"
}

module "common" {
  source = "git::ssh://git@git-aws.internal.justin.tv/esports/terraform.git//modules/variables/twitch-esports-aws/"
}

data "terraform_remote_state" "prod" {
  backend = "s3"
  config {
    bucket = "twitch-esports-aws"
    key = "tfstate/esports/terraform/aws/raw"
    profile = "twitch-esports-aws"
    region = "us-east-1"
  }
}

resource "aws_ecr_repository" "rlcs" {
  name = "rlcs-client"
}

data "template_file" "task_definition" {
  template = "${file("task-definitions/rlcs.json")}"
  vars {
    name = "rlcs-client"
    region = "us-west-2"
    image = "${replace(aws_ecr_repository.rlcs.repository_url, "https://", "")}:latest"
    container_port = 80
    cpu = 256
    mem = 256
  }
}

resource "aws_ecs_task_definition" "default" {
  family = "rlcs-client"
  container_definitions = "${data.template_file.task_definition.rendered}"
}

module "service" {
  source = "git::ssh://git@git-aws.internal.justin.tv/esports/terraform.git//modules/service?ref=00bcb20fa153152c811b6948d83a7085eaaf50b9"
  name = "rlcs-client"
  prefix = "rlcs_client"
  container_port = "80"
  alb = true
  env = "production"
  cluster = "programs"
  # rlcs.gg us-west-2
  certificate_arn = "arn:aws:acm:us-west-2:954268612818:certificate/bc6ef236-5ba0-4fd8-abe5-1eb774bdb559"
  task_definition = "${aws_ecs_task_definition.default.arn}"
  vpc_id = "${lookup(data.terraform_remote_state.prod.ecs_cluster_tourney, "vpc_id")}"
  subnet_ids = "${split(",", lookup(data.terraform_remote_state.prod.ecs_cluster_tourney, "subnet_ids"))}"
  security_group_id = "${lookup(data.terraform_remote_state.prod.ecs_cluster_tourney, "security_group_id")}"
  internal_zone_id = "${data.terraform_remote_state.prod.route53_zone_services_internal}"
  external_zone_id = "${data.terraform_remote_state.prod.route53_zone_services_external}"
}

resource "aws_route53_record" "production" {
  zone_id = "Z140IZPCS2HRJW" # rlcs.gg
  name = ""
  type = "A"

  alias {
    name = "${module.service.alb_dns_name}"
    zone_id = "${module.service.alb_zone_id}"
    evaluate_target_health = false
  }
}

resource "aws_route53_record" "rocketleagueesports" {
  zone_id = "Z34P3OL4J5VSVE" # rocketleagueesports.com
  name = ""
  type = "A"

  alias {
    name = "${module.service.alb_dns_name}"
    zone_id = "${module.service.alb_zone_id}"
    evaluate_target_health = false
  }
}

output "ecr_url" {
  value = "${aws_ecr_repository.rlcs.repository_url}"
}
