import React from 'react';
import _ from 'lodash';
import moment from 'moment-timezone';

import {imageUrl, backgroundImageUrl} from '../common/helpers/image';
import { ALL_REGIONS, REGIONS } from '../common/constants';
import * as style from './matches/styles.scss';

const NUM_WEEKS = 6;
const MATCH_STATUSES = {
  PENDING: 'Pending',
  LIVE: 'Live',
  COMPLETED: 'Completed',
};

export default class MatchesComponent extends React.Component {
  constructor(props){
    super(props);
    this.contents = [];
    this.state = {
      page: 0
    };
  }

  // once both teams/matches are filled, we can use them
  componentDidUpdate() {
    const { teams, matches } = this.props;
    if ( _.isEmpty(this.contents) && !(_.isEmpty(teams) || _.isEmpty(matches))) {
      this.setContents();
    }
  }

  setContents(){
    const { teams, matches } = this.props;

    let results = {};
    let pages = [];
    _.range(1, NUM_WEEKS + 1).forEach((week) => {
      results[week] = {};
      ALL_REGIONS.forEach((region) => {
        const filtered = _.values(matches).filter((match) => {
          return region.key === match.region && week == match.week;
        });
        const sortedResults = _.sortBy(filtered, (match) => new Date(match.date_time));
        results[week][region.key] = sortedResults;
        pages.push({
          week,
          region: region.key,
          date: sortedResults[0] ? sortedResults[0].date_time : null,
          matches: sortedResults,
        });
      });
    });

    // figure out which page to show
    // Keep track of completed pages, if we find a partially complete day, use that page.
    // otherwise use date to show results or show upcoming matches
    let lastCompletedPage = 0;
    let showPage = null; // the more definitive if we find a live game or partial played matches
    let idx = 0;
    for (let page of pages) {
      if(page.matches.length === 0) break;
      const allCompleted = _.every(page.matches, (match) => match.status === MATCH_STATUSES.COMPLETED);
      const containsLive = _.find(page.matches, (match) => match.status === MATCH_STATUSES.LIVE) != null;
      const containsCompleted =  _.some(page.matches, (match) => match.status === MATCH_STATUSES.COMPLETED);
      const containsPending =  _.some(page.matches, (match) => match.status === MATCH_STATUSES.PENDING);
      if(allCompleted){
        lastCompletedPage = idx;
      } else if (containsLive || ( containsCompleted && containsPending )) {
        showPage = idx;
        break;
      } else {
        break;
      }
      idx++;
    }

    let pageForState;
    if(showPage){
      pageForState = showPage;
    } else {
      pageForState = lastCompletedPage;

      // we will default that we show results ^
      // but we still want to check if there are games scheduled
      // and it is < 3 days from now, we will show those.
      const nextPage = pages[lastCompletedPage + 1];
      if (nextPage && nextPage.matches.length > 0) {
        const nextMatchTime = moment.tz(nextPage.matches[0].date_time, "America/Los_Angeles");
        const now = moment();
        const diff = nextMatchTime.diff(now, 'days');
        // console.log('difference', diff);
        if(diff < 3){
          pageForState = lastCompletedPage + 1;
        }
      }
    }

    this.setState({
      page: pageForState
    });
    // console.log('last completed page is', lastCompletedPage);
    // console.log('show Page', showPage);

    this.contents = pages;
  }

  render() {
    const bg = require('./matches/match_bg.jpg');
    const containerStyles = backgroundImageUrl(bg);

    return (
      <div className="section-container matches-section" style={containerStyles}>
         {this.renderCurrentPage()}
         {this.renderControls()}
      </div>
    );
  }

  hasPrevPage() {
    const checkPage = this.state.page - 1;
    return checkPage >= 0;
  }

  prevPage() {
    if (this.hasPrevPage()) {
      this.setState({
        page: this.state.page - 1
      });
    }
  }

  hasNextPage() {
    const checkPage = this.state.page + 1;
    return this.contents[checkPage] != null && this.contents[checkPage].matches.length > 0;
  }

  nextPage() {
    if (this.hasNextPage()) {
      this.setState({
        page: this.state.page + 1
      });
    }
  }

  renderControls() {
    const prevClasses = ['prev-button', this.hasPrevPage() ? 'active' : 'disabled'];
    const nextClasses = ['next-button', this.hasNextPage() ? 'active' : 'disabled'];

    return (
      <div className='match-controls'>
        <div className={prevClasses.join(' ')} onClick={this.prevPage.bind(this)}>&lt; PREV</div>
        <div className={nextClasses.join(' ')} onClick={this.nextPage.bind(this)}>NEXT &gt;</div>
      </div>
    );
  }

  renderCurrentPage() {
    if (_.isEmpty(this.contents)) {
      return null;
    }

    const data = this.contents[this.state.page];
    const matches = data.matches.map((match) => this.renderSingleMatchup(match));

    return(
      <div className="matches-container">
        <div className="week-region">WEEK {data.week} - {REGIONS[data.region].title}</div>
        <div className="date">{moment(data.date).format('LL')}</div>
        <div className="matches-contents">
          {matches}
        </div>
      </div>
    );
  }

  renderSingleMatchup(match) {
    const { teams } = this.props;

    return (
      <div className="single-matchup" key={match.id}>
        <div className="left-team">
          <div className="team-name-abbr">{teams[match.team_one.id].abbr}</div>
          <div className="team-name-full">{teams[match.team_one.id].name}</div>
          <div className="team-score">{match.team_one.score}</div>
        </div>
        <div className="match-details">
          {this.renderStatusContent(match)}
        </div>
        <div className="right-team">
          <div className="team-score">{match.team_two.score}</div>
          <div className="team-name-full">{teams[match.team_two.id].name}</div>
          <div className="team-name-abbr">{teams[match.team_two.id].abbr}</div>
        </div>
      </div>
    );
  }

  renderStatusContent(match){
    const vsContent = <div className="vs">vs.</div>;
    switch(match.status) {
      case MATCH_STATUSES.PENDING:
        return (<div className="pending-container">
          {vsContent}
          <div className="pending">
            {moment.tz(match.date_time, "America/Los_Angeles").tz(moment.tz.guess()).format('h:mm a z')}
          </div>
        </div>);
      case MATCH_STATUSES.LIVE:
        return (<div className="live-container">
          {vsContent}
          <div className="live">Live</div>
        </div>);
      case MATCH_STATUSES.COMPLETED:
        return vsContent;
      default:
        return null;
    }
  }
}
