import React from 'react';
import * as style from './nav/styles.scss';
import { connect } from 'react-redux';
import $ from 'jquery'
import { SECTIONS, TWITCH_NAV_ITEM } from '../common/constants';
import { getUrl } from '../common/helpers';

let sectionsOrder = [
  SECTIONS.MAIN,
  SECTIONS.NEWS,
  SECTIONS.FORMAT,
  SECTIONS.MATCHES,
  SECTIONS.STANDINGS,
];

export class Node extends React.Component {
  render() {
    let { section, selectSection } = this.props;
    const src = require(`../../static/${section.icon}`) ;
    let classes = ['nav-node'];
    if (!section.enabled) {
      classes.push('disabled');
    }
    if (this.props.active) {
      classes.push('active');
    }
    if (section.additionalClasses != null) {
      classes = classes.concat(section.additionalClasses);
    }

    return (
      <li onClick={selectSection} className={classes.join(' ')}>
        <div className="icon"><img src={src} /></div>
        <div className="title">{section.title}</div>
      </li>
    );
  }
}

export class NavComponent extends React.Component {
  constructor(props, context){
    super(props, context);
    // this will be filled out getLayoutContent in cdm
    this.navBreakpoints = []; 
    this.screenHeight = null;
    this.isAutoScrolling = false;

    this.state = {
      activeSection: null,
      navOpen: false        //mobile nav
    };
  }

  componentDidMount() {
    $(window).on('scroll', this.checkNavUpdate.bind(this));
    $(window).on('resize', this.getLayoutContent.bind(this));

    // for initial layout
    setTimeout(this.getLayoutContent.bind(this), 0);
  }

  componentWillUnmount() {
    $(window).off('scroll');
    $(window).off('resize');
  }

  componentWillUpdate(nextProps, nextState) {
    // if live/not-live update sections and navigation 
    if(this.props.isLive !== nextProps.isLive){
      this.checkNavUpdate();
    }

    if(this.props.news.loading !== nextProps.news.loading && getUrl() != null){
      setTimeout(() => {
        this.scrollToId(getUrl().slug);
      }, 0);
    }
  }
  
  toggleNav() {
    this.setState({
      navOpen: !this.state.navOpen
    });
  }

  closeNav(){
    this.setState({
      navOpen: false
    });
  }

  checkNavUpdate(){
    if(this.isAutoScrolling) return false;
    let scrollPos = $(window).scrollTop();
    let breakpointCheck = scrollPos + this.screenHeight/2;
    let result;
    for (let sec of this.navBreakpoints) {
      if(breakpointCheck >= sec.top){
        result = sec.section;
        break;
      }
    }
    this.setState({
      activeSection: result
    });
  }

  scrollToId(id){
    const HEADING_SPACE = 200;
    let destination = $(`#${id}`).offset().top - HEADING_SPACE;
    // $('body, html').scrollTop(destination);
    $('body, html').animate({scrollTop: destination}, 500);
  }

  scrollToSection(section){
    this.isAutoScrolling = true;
    let destination = $(`.${section.title}-section`).offset().top;
    $('body, html').animate({scrollTop: destination}, 500);
    setTimeout(() => {
      this.isAutoScrolling = false;
    }, 500);
    this.setState({
      activeSection: section
    });
  }

  // Getting layout content will manage the size each of the sections,
  // this way we can accurately switch the nav based on scroll location
  getLayoutContent(){
    this.screenHeight = $(window).height();
    let sections = sectionsOrder.filter((sec) => sec.enabled && !sec.externalLink);
    if(!this.props.isLive){
      sections = [...sections, SECTIONS.WATCH];
    }
    this.navBreakpoints = sections.map((sec, i) => {
      return {
        section: sec,
        top: $(`.${sec.title}-section`).offset().top
      };
    }).reverse();
    this.checkNavUpdate();
  }

  setSection(sec){
    if (!sec.enabled) return false;
    this.scrollToSection(sec);
    this.closeNav();
  }

  externalLink(url) {
    window.location = url;
  }

  render() {
    let selected = SECTIONS.MAIN;
    let sections = sectionsOrder.concat([TWITCH_NAV_ITEM]);
    let logo = require('../../static/logo.svg');
    let navClasses = ['navigation'];
    let mobileIcon = ['fa'];
    if (this.state.navOpen){
      navClasses.push('open');
      mobileIcon.push('fa-times');
    }else{
      mobileIcon.push('fa-bars');
    }

    if(!this.props.isLive){
      sections = [...sections, SECTIONS.WATCH];
    }
    sections = [...sections, SECTIONS.RULES];

    let allSections = sections.map((sec, i) => {
      let action;
      if (sec.externalLink) {
        action = this.externalLink.bind(this, sec.externalLink);
      } else {
        action = this.setSection.bind(this, sec);
      }

      let props = {
        key: sec.title,
        section: sec,
        active: this.state.activeSection === sec,
        selectSection: action
      };
      return <Node {...props} />;
    });

    return (
      <nav className={navClasses.join(' ')}>
        <div className="nav-toggle" onClick={this.toggleNav.bind(this)}>
          <i className={mobileIcon.join(' ')} />
        </div>
        <img className="nav-logo" src={logo} />
        <ul className="nav-contents">
          {allSections}
        </ul>
      </nav>
    );
  }
}

export default connect(state => state)(NavComponent);
