provider "aws" {
    region = "us-west-2"
    profile = "twitch-esports-dev"
}

module "common" {
  source = "git::ssh://git@git-aws.internal.justin.tv/esports/terraform.git//modules/variables/twitch-esports-dev/"
}

variable "service_name" { default = "rlcs" }
variable "service_env" { default = "staging" }

data "terraform_remote_state" "dev" {
  backend = "s3"
  config {
    bucket = "twitch-esports-dev"
    key = "tfstate/esports/terraform/dev/raw"
    profile = "twitch-esports-dev"
    region = "us-east-1"
  }
}

resource "aws_s3_bucket" "default" {
  bucket = "${var.service_name}-${var.service_env}"
  website {
    index_document = "index.html"
    error_document = "error.html"
  }
  policy = <<EOF
{
  "Version":"2012-10-17",
  "Statement":[
    {
      "Sid":"AddPerm",
      "Effect":"Allow",
      "Principal": "*",
      "Action":["s3:GetObject"],
      "Resource":["arn:aws:s3:::${var.service_name}-${var.service_env}/*"]
    }
  ]
}
EOF
}

resource "aws_db_subnet_group" "default" {
  name = "${var.service_name}"
  subnet_ids = ["${split(",", lookup(data.terraform_remote_state.dev.ecs_cluster_programs, "subnet_ids"))}"]
}

resource "aws_security_group" "db" {
  name = "${var.service_name}-db"
  description = "RDS for ${var.service_name}"
  vpc_id = "${lookup(data.terraform_remote_state.dev.ecs_cluster_programs, "vpc_id")}"
  ingress {
    from_port = "3306"
    to_port = "3306"
    protocol = "tcp"
    cidr_blocks = ["10.0.0.0/8"]
  }
  egress {
    from_port = 0
    to_port = 0
    protocol = "-1"
    cidr_blocks = ["10.0.0.0/8"]
  }
}

resource "aws_db_instance" "default" {
  allocated_storage = 15
  engine = "mysql"
  engine_version = "5.6.27"
  instance_class = "db.t2.micro"
  name = "${var.service_name}"
  username = "${var.service_name}"
  password = "zpSD9zlBo2bI"
  db_subnet_group_name = "${aws_db_subnet_group.default.name}"
  vpc_security_group_ids = ["${aws_security_group.db.id}"]
}

resource "aws_ecr_repository" "frontend" {
  name = "${var.service_name}-frontend"
}

resource "aws_ecr_repository" "backend" {
  name = "${var.service_name}-backend"
}

resource "aws_iam_role" "task" {
  name = "ecs_${var.service_name}_task_role"
  assume_role_policy = <<EOF
{
  "Version": "2008-10-17",
  "Statement": [{
    "Action": "sts:AssumeRole",
    "Principal": {
      "Service": [
        "ecs-tasks.amazonaws.com"
      ]
    },
    "Effect": "Allow"
  }]
}
EOF
}

resource "aws_iam_role_policy" "task_s3" {
  name = "ecs_${var.service_name}_task_role_s3"
  role = "${aws_iam_role.task.id}"
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": ["s3:*"],
      "Resource": ["${aws_s3_bucket.default.arn}"]
    },
    {
      "Effect": "Allow",
      "Action": ["s3:*"],
      "Resource": ["${aws_s3_bucket.default.arn}/*"]
    }
  ]
}
EOF
}

resource "template_file" "default" {
  template = "${file("task-definitions/rlcs.json")}"
  vars {
    name = "${var.service_name}"
    region = "us-west-2"
    env = "${var.service_env}"
    image_frontend = "${replace(aws_ecr_repository.frontend.repository_url, "https://", "")}:latest"
    image_backend = "${replace(aws_ecr_repository.backend.repository_url, "https://", "")}:latest"
    db_url = "mysql://${aws_db_instance.default.username}:${aws_db_instance.default.password}@${aws_db_instance.default.endpoint}/${aws_db_instance.default.name}"
    db_hostname = "${aws_db_instance.default.address}"
    db_port = "${aws_db_instance.default.port}"
    db_username = "${aws_db_instance.default.username}"
    db_password = "${aws_db_instance.default.password}"
    db_name = "${aws_db_instance.default.name}"
    port_frontend = 80
    port_backend = 8080
    cpu = 256
    mem = 256
  }
}

resource "aws_ecs_task_definition" "default" {
  family = "${var.service_name}"
  container_definitions = "${template_file.default.rendered}"
  task_role_arn = "${aws_iam_role.task.arn}"
}

module "service" {
  source = "git::ssh://git@git-aws.internal.justin.tv/esports/terraform.git//modules/service?ref=2017-05-18"
  name = "${var.service_name}"
  prefix = "${replace(var.service_name, "-", "_")}"
  container_port = "80"
  env = "staging"
  alb = true
  nginx = true
  desired_count = 4
  certificate_arn = "${module.common.service_certificate_external}"
  task_definition = "${aws_ecs_task_definition.default.arn}"
  vpc_id = "${lookup(data.terraform_remote_state.dev.ecs_cluster_programs, "vpc_id")}"
  cluster = "programs"
  subnet_ids = "${split(",", lookup(data.terraform_remote_state.dev.ecs_cluster_programs, "subnet_ids"))}"
  security_group_id = "${lookup(data.terraform_remote_state.dev.ecs_cluster_programs, "security_group_id")}"
  internal_zone_id = "${data.terraform_remote_state.dev.route53_zone_services_internal}"
  external_zone_id = "${data.terraform_remote_state.dev.route53_zone_services_external}"
  alarm_sns_topic_arn = "${data.terraform_remote_state.dev.alarm_sns_topic_arn}"
}

output "ecr_frontend_url" {
  value = "${aws_ecr_repository.frontend.repository_url}"
}

output "ecr_backend_url" {
  value = "${aws_ecr_repository.backend.repository_url}"
}

