resource "aws_key_pair" "ecs_default" {
  key_name = "ecs_cluster_${var.name}"
  public_key = "${file(var.ssh_pubkey)}"
}

resource "aws_autoscaling_group" "ecs_cluster" {
  availability_zones = ["${var.availability_zones}"]
  name = "ecs_${var.name}"
  min_size = "${var.autoscale_min}"
  max_size = "${var.autoscale_max}"
  desired_capacity = "${var.autoscale_desired}"
  health_check_type = "EC2"
  launch_configuration = "${aws_launch_configuration.ecs.name}"
  vpc_zone_identifier = ["${var.subnet_ids}"]
  termination_policies = ["OldestLaunchConfiguration", "Default"]

  tag {
    key = "Owner"
    value = "marian@twitch.tv"
    propagate_at_launch = true
  }

  tag {
    key = "Environment"
    value = "${var.environment}"
    propagate_at_launch = true
  }

  tag {
    key = "Service"
    value = "esports/${var.name}/ecs"
    propagate_at_launch = true
  }
}

resource "aws_launch_configuration" "ecs" {
  # We need to auto-generate the name to update AMI/etc., see:
  # https://github.com/hashicorp/terraform/issues/2958
  # TODO prefix computed value w/ name?
  # name = "ECS ${var.name}"
  image_id = "${lookup(var.amis, var.region)}"
  instance_type = "${var.instance_type}"
  security_groups = ["${aws_security_group.ecs.id}"]
  iam_instance_profile = "${aws_iam_instance_profile.ecs.name}"
  # TODO: is there a good way to make the key configurable sanely?
  key_name = "${aws_key_pair.ecs_default.key_name}"
  associate_public_ip_address = true
  user_data = "#!/bin/bash\necho ECS_CLUSTER='${var.name}' > /etc/ecs/ecs.config"

  lifecycle {
    create_before_destroy = true
  }
}
