provider "aws" {
    region = "us-east-1"
    profile = "twitch-esports-aws"
}

module "common" {
  source = "../../../modules/variables/twitch-esports-aws/"
}

data "terraform_remote_state" "aws" {
  backend = "s3"
  config {
    bucket = "twitch-esports-aws"
    key = "tfstate/esports/terraform/aws/raw"
    profile = "twitch-esports-aws"
    region = "us-east-1"
  }
}

resource "aws_ecr_repository" "rlcs-frontend" {
  name = "rlcs-frontend"
}

resource "template_file" "frontend_task_definition" {
  template = "${file("task-definitions/rlcs-frontend.json")}"
  vars {
    name = "rlcs-frontend"
    image = "${replace(aws_ecr_repository.rlcs-frontend.repository_url, "https://", "")}:latest"
    container_port = 80
    cpu = 256
    mem = 256
  }
}

resource "aws_route53_record" "production" {
  zone_id = "Z34P3OL4J5VSVE" # rocketleagueesports.com
  name = ""
  type = "A"

  alias {
    name = "${module.service.alb_dns_name}"
    zone_id = "${module.service.alb_zone_id}"
    evaluate_target_health = false
  }
}

resource "aws_ecs_task_definition" "frontend_task" {
  family = "rlcs-frontend"
  container_definitions = "${template_file.frontend_task_definition.rendered}"
}

module "service" {
  source = "../../../modules/service"
  name = "rlcs-frontend"
  prefix = "rlcs_frontend"
  container_port = "80"
  desired_count = 4
  env = "production"
  certificate_arn = "arn:aws:acm:us-east-1:954268612818:certificate/b915615e-9cdb-4d6c-b546-b6476327e813" # rocketleagueesports.com
  task_definition = "${aws_ecs_task_definition.frontend_task.arn}"
  vpc_id = "${lookup(data.terraform_remote_state.aws.ecs_cluster_programs, "vpc_id")}"
  cluster_name = "${lookup(data.terraform_remote_state.aws.ecs_cluster_programs, "name")}"
  cluster_id = "${lookup(data.terraform_remote_state.aws.ecs_cluster_programs, "id")}"
  subnet_ids = "${split(",", lookup(data.terraform_remote_state.aws.ecs_cluster_programs, "subnet_ids"))}"
  security_group_id = "${lookup(data.terraform_remote_state.aws.ecs_cluster_programs, "security_group_id")}"
  internal_zone_id = "${data.terraform_remote_state.aws.route53_zone_services_internal}"
  external_zone_id = "${data.terraform_remote_state.aws.route53_zone_services_external}"
}

output "ecr_frontend_url" {
  value = "${aws_ecr_repository.rlcs-frontend.repository_url}"
}
