provider "aws" {
    region = "us-west-2"
    profile = "twitch-esports-dev"
}

module "common" {
  source = "../../../modules/variables/twitch-esports-dev/"
}

data "terraform_remote_state" "dev" {
  backend = "s3"
  config {
    bucket = "twitch-esports-dev"
    key = "tfstate/esports/terraform/dev/raw"
    profile = "twitch-esports-dev"
    region = "us-east-1"
  }
}

resource "aws_ecr_repository" "rlcs-frontend" {
  name = "rlcs-frontend"
}

resource "template_file" "frontend_task_definition" {
  template = "${file("task-definitions/rlcs-frontend.json")}"
  vars {
    name = "rlcs-frontend"
    image = "${replace(aws_ecr_repository.rlcs-frontend.repository_url, "https://", "")}:latest"
    container_port = 80
    cpu = 256
    mem = 256
  }
}

resource "aws_ecs_task_definition" "frontend_task" {
  family = "rlcs-frontend"
  container_definitions = "${template_file.frontend_task_definition.rendered}"
}

module "service" {
  source = "../../../modules/service"
  name = "rlcs-frontend"
  prefix = "rlcs_frontend"
  container_port = "80"
  env = "staging"
  certificate_arn = "${module.common.service_certificate_external}"
  task_definition = "${aws_ecs_task_definition.frontend_task.arn}"
  vpc_id = "${lookup(data.terraform_remote_state.dev.ecs_cluster_programs, "vpc_id")}"
  cluster_name = "${lookup(data.terraform_remote_state.dev.ecs_cluster_programs, "name")}"
  cluster_id = "${lookup(data.terraform_remote_state.dev.ecs_cluster_programs, "id")}"
  subnet_ids = "${split(",", lookup(data.terraform_remote_state.dev.ecs_cluster_programs, "subnet_ids"))}"
  security_group_id = "${lookup(data.terraform_remote_state.dev.ecs_cluster_programs, "security_group_id")}"
  internal_zone_id = "${data.terraform_remote_state.dev.route53_zone_services_internal}"
  external_zone_id = "${data.terraform_remote_state.dev.route53_zone_services_external}"
}

output "ecr_frontend_url" {
  value = "${aws_ecr_repository.rlcs-frontend.repository_url}"
}
