// +build test

package handlers

import (
	"fmt"
	"testing"

	"ting/model"
	. "ting/util/types"
)

func (ht handlerTest) PostAnswerTests(t *testing.T) {
	makeStuff := func(t *testing.T, qAttr model.Attr) (u *model.User, q *model.Question, a *model.Answer, url string) {
		u = ht.db.NewTestUser(t, nil)
		q = ht.db.NewTestQuestion(t, qAttr)
		a = q.Answers[0]
		url = fmt.Sprintf("/answer?answer_id=%d&user_id=%s", a.ID, u.OpaqueID)
		return
	}

	t.Run("200", func(t *testing.T) {
		u, q, _, url := makeStuff(t, model.Attr{"Active": true})
		code, body := ht.POST(t, url, "")
		if code != 200 {
			t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
		}
		ua := model.ParseUserAnswer(t, body)
		var err model.Error
		if err = q.ForUser(u.OpaqueID); err != nil {
			t.Fatalf("unexpected error re-fetching question with UserAnswer: %s", err.Error())
		} else if q.UserAnswer == nil {
			t.Fatal("unexpected nil for question's UserAnswer")
		}
		ua.AssertEqual(t, q.UserAnswer)
	})

	t.Run("NoAnswer", func(t *testing.T) {
		url := fmt.Sprintf("/answer?answer_id=-1&user_id=1")
		if code, body := ht.POST(t, url, ""); code != 404 {
			t.Fatalf("wrong HTTP status code: %d; expected 404\nbody: %s", code, body)
		} else if reply := parseErrReply(t, body); reply.Error != "no question with this answer" {
			t.Fatalf("wrong error message: %q", reply.Error)
		}
	})

	t.Run("Inactive", func(t *testing.T) {
		_, q, _, url := makeStuff(t, nil)
		expErr := fmt.Sprintf("question %d has never been active", q.ID)
		if code, body := ht.POST(t, url, ""); code != 400 {
			t.Fatalf("wrong HTTP status code: %d; expected 400\nbody: %s", code, body)
		} else if reply := parseErrReply(t, body); reply.Error != expErr {
			t.Fatalf("wrong error message: %q", reply.Error)
		}
	})

	t.Run("Duplicate", func(t *testing.T) {
		_, _, _, url := makeStuff(t, model.Attr{"Active": true})
		if code, body1 := ht.POST(t, url, ""); code != 200 {
			t.Fatalf("unexpected error for first answer: HTTP %d\nbody: %s", code, body1)
		} else if code, body2 := ht.POST(t, url, ""); code != 200 {
			t.Fatalf("unexpected error for second answer: HTTP %d\nbody: %s", code, body2)
		} else {
			ua1 := model.ParseUserAnswer(t, body1)
			ua2 := model.ParseUserAnswer(t, body2)
			ua1.AssertEqual(t, ua2)
		}
	})

	t.Run("Change", func(t *testing.T) {
		u, q, _, url := makeStuff(t, model.Attr{"Active": true})
		if code, body := ht.POST(t, url, ""); code != 200 {
			t.Fatalf("unexpected error for first request: HTTP %d\nbody: %s", code, body)
		}

		url = fmt.Sprintf("/answer?answer_id=%d&user_id=%s", q.AnswerIDs[1], u.OpaqueID)
		if code, body := ht.POST(t, url, ""); code != 400 {
			t.Fatalf("wrong HTTP status code for second request: %d; expected 400\nbody: %s", code, body)
		} else if reply := parseErrReply(t, body); reply.Error != "user has already answered this question" {
			t.Fatalf("wrong error message: %q", reply.Error)
		}
	})

	t.Run("JWT", func(t *testing.T) {
		u, q, _, url := makeStuff(t, model.Attr{"Active": true})
		jwtAttrs := StringMap{"opaque_user_id": u.OpaqueID, "channel_id": q.ChannelID}
		url = "/jwt" + url

		t.Run("WrongUser", func(t *testing.T) {
			headers := makeJWTHeader(t, nil, jwtAttrs.With("opaque_user_id", "lol"))
			code, body, _ := request(t, ht.engine, "POST", url, "", headers)
			if code != 403 {
				t.Fatalf("wrong HTTP status code: %d; expected 403", code)
			} else if reply := parseErrReply(t, body); reply.Error != "cannot vote on behalf of another user" {
				t.Fatalf("wrong error message: %q", reply.Error)
			}
		})

		t.Run("WrongChannel", func(t *testing.T) {
			headers := makeJWTHeader(t, nil, jwtAttrs.With("channel_id", -1))
			code, body, _ := request(t, ht.engine, "POST", url, "", headers)
			if code != 403 {
				t.Fatalf("wrong HTTP status code: %d; expected 403", code)
			} else if reply := parseErrReply(t, body); reply.Error != "cannot post answer to a different channel" {
				t.Fatalf("wrong error message: %q", reply.Error)
			}
		})

		t.Run("OK", func(t *testing.T) {
			headers := makeJWTHeader(t, nil, jwtAttrs)
			code, body, _ := request(t, ht.engine, "POST", url, "", headers)
			if code != 200 {
				t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
			}
			ua := model.ParseUserAnswer(t, body)
			var err model.Error
			if err = q.ForUser(u.OpaqueID); err != nil {
				t.Fatalf("unexpected error re-fetching question with UserAnswer: %s", err.Error())
			} else if q.UserAnswer == nil {
				t.Fatal("unexpected nil for question's UserAnswer")
			}
			ua.AssertEqual(t, q.UserAnswer)
		})
	})
}
