package handlers

import (
	"ting/model"

	"github.com/gin-gonic/gin"
)

func addChannelHandlers(engine gin.IRouter, db *model.DB) {
	engine.OPTIONS("/channels/:channel_id", makeOptionsHandler("GET", "PATCH"))

	engine.GET("/channels/:channel_id", GetChannelHandler(db))
	engine.PATCH("/channels/:channel_id", PatchChannelHandler(db))
}

// GET /channels/:channel_id
func GetChannelHandler(db *model.DB) gin.HandlerFunc {
	h := NewHandler(DefParam("channel_id", "int", "url", true, nil))
	return h.Wrap(func(ctx *gin.Context) {
		channelID := h.IntParam("channel_id")
		opaqueID := ""
		if _, found := ctx.Get("jwt"); found {
			if ctx.GetInt("channel_id") != channelID {
				ctx.JSON(403, gin.H{"error": "channel info must be requested from that channel"})
				return
			}
			opaqueID = ctx.GetString("opaque_id")
		}

		if channel, err := db.FindOrCreateChannel(channelID); err != nil {
			replyError(ctx, err)
		} else if err = channel.WithCurrentQuestion(); err != nil {
			replyError(ctx, err)
		} else if opaqueID == "" {
			ctx.JSON(200, channel)
		} else if err = channel.ForUser(opaqueID); err != nil {
			replyError(ctx, err)
		} else {
			ctx.JSON(200, channel)
		}
	})
}

// PATCH /channels/:channel_id
func PatchChannelHandler(db *model.DB) gin.HandlerFunc {
	h := NewHandler(DefParam("channel_id", "int", "url", true, nil))

	return h.Wrap(func(ctx *gin.Context) {
		channelID := h.IntParam("channel_id")
		var updates map[string]interface{}
		if jwtRole, found := ctx.Get("role"); found && jwtRole != "broadcaster" {
			ctx.JSON(403, gin.H{"error": "only broadcaster can modify their own channel"})
		} else if jwtChannelID, found := ctx.Get("channel_id"); found && jwtChannelID != channelID {
			ctx.JSON(403, gin.H{"error": "cannot modify another broadcaster's channel"})
		} else if bindErr := ctx.BindJSON(&updates); bindErr != nil {
			ctx.JSON(400, gin.H{"error": "invalid JSON"})
		} else if c, err := db.FindOrCreateChannel(channelID); err != nil {
			replyError(ctx, err)
		} else if result, err := c.Update(updates); err != nil {
			replyError(ctx, err)
		} else {
			ctx.JSON(200, result)
		}
	})
}
