// +build test

package handlers

import (
	"fmt"
	"testing"

	"ting/model"
	"ting/util"
	. "ting/util/types"
)

func (ht handlerTest) GetChannelTests(t *testing.T) {
	t.Run("NewChannel", func(t *testing.T) {
		channelID := ht.db.NewTestChannel(t, nil).ID
		if _, err := ht.db.Exec(`DELETE FROM "channels" WHERE "id" = $1`, channelID); err != nil {
			t.Fatalf("error deleting new channel %d: %s", channelID, err)
		}

		url := fmt.Sprintf("/channels/%d", channelID)
		code, body := ht.GET(t, url)
		if code != 200 {
			t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
		}

		cActual := model.ParseChannel(t, body)
		cExpected := model.ParseChannel(t, `{"id": %d}`, channelID)
		cActual.AssertEqual(t, cExpected)
	})

	t.Run("ExistingChannel", func(t *testing.T) {
		channelID := ht.db.NewTestChannel(t, nil).ID

		url := fmt.Sprintf("/channels/%d", channelID)
		code, body := ht.GET(t, url)
		if code != 200 {
			t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
		}

		cActual := model.ParseChannel(t, body)
		cExpected := model.ParseChannel(t, `{"id": %d}`, channelID)
		cActual.AssertEqual(t, cExpected)
	})

	t.Run("WithCurrentQuestion", func(t *testing.T) {
		question := ht.db.NewTestQuestion(t, model.Attr{"Active": true})

		url := fmt.Sprintf("/channels/%d", question.ChannelID)
		code, body := ht.GET(t, url)
		if code != 200 {
			t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
		}

		cActual := model.ParseChannel(t, body)
		cExpected := model.ParseChannel(t,
			`{"id": %d, "current_question_id": %d, "current_question": %s}`,
			question.ChannelID, question.ID, util.ToJSON(t, question))
		cActual.AssertEqual(t, cExpected)
	})

	t.Run("JWT", func(t *testing.T) {
		t.Run("WrongID", func(t *testing.T) {
			headers := makeJWTHeader(t, nil, StringMap{"channel_id": 1})
			code, body, _ := request(t, ht.engine, "GET", "/jwt/channels/2", "", headers)
			if code != 403 {
				t.Fatalf("wrong HTTP response code: %d\nbody: %s", code, body)
			} else if reply := parseErrReply(t, body); reply.Error != "channel info must be requested from that channel" {
				t.Fatalf("wrong error message: %q\n", reply.Error)
			}
		})

		makeStuff := func(t *testing.T, iAnswer int) (c *model.Channel, headers map[string]string, url string) {
			t.Helper()
			u := ht.db.NewTestUser(t, nil)
			c = ht.db.NewTestChannel(t, nil)
			if iAnswer != -1 {
				ht.db.NewTestQuestion(t, model.Attr{
					"ChannelID":  c.ID,
					"UserID":     u.ID,
					"UserAnswer": iAnswer,
				})
			}
			headers = makeJWTHeader(t, nil, StringMap{"opaque_user_id": u.OpaqueID, "channel_id": c.ID})
			url = fmt.Sprintf("/jwt/channels/%d", c.ID)
			return
		}

		for i, name := range []string{"NoCurrentQuestion", "NoUserAnswer", "WithUserAnswer"} {
			t.Run(name, func(t *testing.T) {
				c, headers, url := makeStuff(t, i-1)
				code, body, _ := request(t, ht.engine, "GET", url, "", headers)
				if code != 200 {
					t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
				}
				model.ParseChannel(t, body).AssertEqual(t, c)
			})
		}
	})
}

func (ht handlerTest) PatchChannelTests(t *testing.T) {
	reqBodyFmt := `{"id": %d, "settings": {"theme": "t1", "theme_settings": {"k1": "v1", "k2": "v2"}}}`

	t.Run("OK", func(t *testing.T) {
		channelID := ht.db.NewTestChannel(t, nil).ID
		reqBody := fmt.Sprintf(reqBodyFmt, channelID)
		cExp := model.ParseChannel(t, reqBody)

		url := fmt.Sprintf("/channels/%d", channelID)
		code, body := ht.PATCH(t, url, reqBody)
		if code != 200 {
			t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
		}
		model.ParseChannel(t, body).AssertEqual(t, cExp)
	})

	t.Run("JWT", func(t *testing.T) {
		channelID := ht.db.NewTestChannel(t, nil).ID
		jwtAttrs := StringMap{"role": "broadcaster", "channel_id": channelID}
		reqBody := fmt.Sprintf(reqBodyFmt, channelID)
		url := fmt.Sprintf("/jwt/channels/%d", channelID)

		t.Run("WrongRole", func(t *testing.T) {
			headers := makeJWTHeader(t, nil, jwtAttrs.With("role", "moderator"))
			code, body, _ := request(t, ht.engine, "PATCH", url, reqBody, headers)
			if code != 403 {
				t.Fatalf("wrong HTTP status code: %d\nbody: %s", code, body)
			} else if reply := parseErrReply(t, body); reply.Error != "only broadcaster can modify their own channel" {
				t.Fatalf("wrong error message: %q", reply.Error)
			}
		})

		t.Run("WrongChannel", func(t *testing.T) {
			headers := makeJWTHeader(t, nil, jwtAttrs.With("channel_id", -1))
			code, body, _ := request(t, ht.engine, "PATCH", url, reqBody, headers)
			if code != 403 {
				t.Fatalf("wrong HTTP status code: %d\nbody: %s", code, body)
			} else if reply := parseErrReply(t, body); reply.Error != "cannot modify another broadcaster's channel" {
				t.Fatalf("wrong error message: %q", reply.Error)
			}
		})

		t.Run("OK", func(t *testing.T) {
			headers := makeJWTHeader(t, nil, jwtAttrs)
			code, body, _ := request(t, ht.engine, "PATCH", url, reqBody, headers)
			if code != 200 {
				t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
			}
			model.ParseChannel(t, body).AssertEqual(t, model.ParseChannel(t, reqBody))
		})
	})
}
