package handlers

import (
	"errors"
	"fmt"
	"strconv"
	"strings"
	"time"

	"ting/model"
	"ting/util/set"

	"github.com/gin-gonic/gin"
)

func Register(engine gin.IRouter, db *model.DB) {
	addChannelHandlers(engine, db)
	addQuestionHandlers(engine, db)
	addAnswerHandlers(engine, db)
	addLeaderboardHandlers(engine, db)
}

func makeOptionsHandler(methods ...string) gin.HandlerFunc {
	methods = append(methods, "OPTIONS")
	methodStr := strings.Join(methods, ", ")
	return func(ctx *gin.Context) {
		ctx.Header("Allow", methodStr)
		ctx.Header("Access-Control-Allow-Methods", methodStr)

		status := 200
		if method := ctx.GetHeader("Access-Control-Request-Method"); method != "" {
			status = 405
			for _, m := range methods {
				if m == method {
					status = 200
					break
				}
			}
		}

		ctx.String(status, "")
	}
}

func replyError(ctx *gin.Context, err model.Error) {
	code := 500
	switch err.Kind() {
	case model.UserErrorKind:
		code = 400
	}
	ctx.JSON(code, err)
}

var paramLocations set.StringSet
var paramTypes set.StringSet

func init() {
	paramLocations = set.NewStringSet("url", "query")
	paramTypes = set.NewStringSet("string", "int", "time")
}

type paramSpec struct {
	name string
	ty   string // recognized values are in `paramTypes`
	loc  string // recognized values are in `paramLocations`
	req  bool
	def  interface{}
}

func DefParam(name, ty, loc string, req bool, def interface{}) paramSpec {
	if !paramLocations.Has(loc) {
		panic(fmt.Sprintf(`invalid parameter location: %q; allowed: %s`, loc, paramLocations.Values()))
	} else if !paramTypes.Has(ty) {
		panic(fmt.Sprintf(`invalid parameter type: %q; allowed: %s`, ty, paramTypes.Values()))
	}
	return paramSpec{
		loc:  loc,
		name: name,
		ty:   ty,
		req:  req,
		def:  def,
	}
}

func (p paramSpec) parse(ctx *gin.Context) (interface{}, error) {
	var vStr string
	var found bool
	if p.loc == "url" {
		vStr, found = ctx.Params.Get(p.name)
	} else {
		vStr, found = ctx.GetQuery(p.name)
	}

	if !found {
		if p.req {
			return nil, errors.New("missing")
		}
		return p.def, nil
	}

	switch p.ty {
	case "string":
		return vStr, nil
	case "int":
		if v, err := strconv.Atoi(vStr); err != nil {
			return nil, errors.New("must be an integer")
		} else {
			return v, nil
		}
	case "time":
		if v, err := time.Parse(time.RFC3339, vStr); err != nil {
			return nil, errors.New("must be timestamp (RFC 3339)")
		} else {
			return &v, nil
		}
	}
	panic(fmt.Sprintf("unrecognized parameter type for %q: %q", p.name, p.ty))
}

type paramHandler struct {
	paramDefs []paramSpec
	handler   gin.HandlerFunc
	params    map[string]interface{}
}

func NewHandler(paramDefs ...paramSpec) *paramHandler {
	return &paramHandler{
		paramDefs: paramDefs,
		handler:   nil,
		params:    nil,
	}
}

func (ph *paramHandler) parseParams(ctx *gin.Context) (errs []string) {
	ph.params = make(map[string]interface{}, len(ph.paramDefs))
	for _, p := range ph.paramDefs {
		if v, err := p.parse(ctx); err != nil {
			errs = append(errs, fmt.Sprintf("%s: %s", p.name, err.Error()))
		} else if v != nil {
			ph.params[p.name] = v
		}
	}
	return errs
}

func (ph paramHandler) Param(name string) interface{} {
	v, _ := ph.params[name]
	return v
}

func (ph paramHandler) OptParam(name string) (interface{}, bool) {
	// Go infers the wrong return type for just `return ph.params[name]`.
	v, found := ph.params[name]
	return v, found
}

func (ph paramHandler) StringParam(name string) string {
	return ph.Param(name).(string)
}

func (ph paramHandler) StringOptParam(name string) (string, bool) {
	if v, found := ph.params[name]; !found {
		return "", false
	} else {
		return v.(string), true
	}
}

func (ph paramHandler) IntParam(name string) int {
	return ph.Param(name).(int)
}

func (ph paramHandler) IntOptParam(name string) (int, bool) {
	if v, found := ph.params[name]; !found {
		return 0, false
	} else {
		return v.(int), true
	}
}

func (ph paramHandler) TimeParam(name string) *time.Time {
	return ph.Param(name).(*time.Time)
}

func (ph paramHandler) TimeOptParam(name string) (*time.Time, bool) {
	if v, found := ph.params[name]; !found {
		return nil, false
	} else {
		return v.(*time.Time), true
	}
}

func (ph *paramHandler) Wrap(f gin.HandlerFunc) gin.HandlerFunc {
	return func(ctx *gin.Context) {
		if errs := ph.parseParams(ctx); len(errs) > 0 {
			ctx.JSON(400, gin.H{"errors": errs})
		} else {
			f(ctx)
		}
	}
}
