// +build test

package handlers

import (
	"encoding/base64"
	"encoding/json"
	"math/rand"
	"reflect"
	"strings"
	"testing"
	"time"
	"unicode"

	"ting/model"

	"github.com/gin-gonic/gin"
)

// Entry point for `ting/handlers` tests.
func TestHandlers(t *testing.T) {
	t.Helper()
	db := model.InitTestDB(t)
	defer db.Close()
	initHandlerTest(t, db).runTests(t)
}

// The rest of this is convenience methods & identifying tests:

func init() {
	rand.Seed(time.Now().UnixNano())
	gin.SetMode(gin.TestMode)
}

type handlerTest struct {
	db     *model.DB
	engine *gin.Engine
}

func initHandlerTest(t *testing.T, db *model.DB) handlerTest {
	engine := gin.New()
	Register(engine, db)

	// Re-register handlers with JWT middleware under "/jwt" prefix.
	jwtSecretB64 := base64.URLEncoding.EncodeToString(jwtSecret)
	if mw, err := JWTMiddleware(jwtSecretB64); err != nil {
		t.Fatalf("error initializing JWT middleware: %s", err)
	} else {
		jwtGroup := engine.Group("/jwt", mw)
		Register(jwtGroup, db)
	}

	return handlerTest{
		db:     db,
		engine: engine,
	}
}

func (ht handlerTest) request(t *testing.T, method, url, reqBody string) (code int, body string) {
	t.Helper()
	code, body, _ = request(t, ht.engine, method, url, reqBody, nil)
	return
}

func (ht handlerTest) GET(t *testing.T, url string) (code int, body string) {
	t.Helper()
	return ht.request(t, "GET", url, "")
}

func (ht handlerTest) POST(t *testing.T, url, reqBody string) (code int, body string) {
	t.Helper()
	return ht.request(t, "POST", url, reqBody)
}

func (ht handlerTest) PATCH(t *testing.T, url, reqBody string) (code int, body string) {
	t.Helper()
	return ht.request(t, "PATCH", url, reqBody)
}

func (ht handlerTest) DELETE(t *testing.T, url string) (code int, body string) {
	t.Helper()
	return ht.request(t, "DELETE", url, "")
}

type errReply struct {
	Error string `json:"error"`

	// `Conflicts` appears in replies to `POST /questions` when there are overlapping active times.
	Conflicts []int `json:"conflicts,omitempty"`
}

func parseErrReply(t *testing.T, body string) errReply {
	t.Helper()
	var reply errReply
	if err := json.Unmarshal([]byte(body), &reply); err != nil {
		t.Fatalf("failed to decode error JSON: %s\nparse error: %s", body, err)
	} else if reply.Error == "" {
		t.Fatalf(`error JSON did not include "error" field: %s`, body)
	}
	return reply
}

var testingTType reflect.Type

func init() {
	testingTType = reflect.TypeOf(new(testing.T))
}

func isTestMethod(m reflect.Method) (string, bool) {
	if strings.HasSuffix(m.Name, "Tests") &&
		unicode.IsUpper(rune(m.Name[0])) &&
		m.Type.NumIn() == 2 && m.Type.NumOut() == 0 &&
		m.Type.In(1) == testingTType {
		return m.Name[:len(m.Name)-len("Tests")], true
	}
	return "", false

}

func (ht handlerTest) runTests(t *testing.T) {
	htT := reflect.TypeOf(ht)
	htV := reflect.ValueOf(ht)
	for i := 0; i < htT.NumMethod(); i++ {
		method := htT.Method(i)
		if testName, ok := isTestMethod(method); ok {
			testFunc := htV.MethodByName(method.Name).Interface().(func(*testing.T))
			t.Run(testName, testFunc)
		}
	}
}
