package handlers

import (
	"encoding/base64"
	"errors"
	"fmt"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/tshadwell/jwt"

	"github.com/gin-gonic/gin"
)

// https://dev.twitch.tv/docs/extensions/reference#jwt-schema
type jwtClaims struct {
	// Stuff we care about:
	Expiration int64 `json:"exp"`
	// ChannelID: Always `int` (at least on backend), but the extension frontend library encodes them as strings.
	ChannelID string `json:"channel_id"`
	OpaqueID  string `json:"opaque_user_id"`
	Role      string `json:"role"`

	// Stuff we maybe don't care about?
	PubsubPerms map[string][]string `json:"pubsub_perms"`
	TwitchID    string              `json:"user_id"`
}

func (c jwtClaims) validate() error {
	if c.Expiration == 0 {
		return errors.New("missing expiration")
	} else if exp := time.Unix(c.Expiration, 0); exp.Before(time.Now()) {
		return errors.New("token expired")
	} else if c.ChannelID == "" {
		return errors.New("missing channel_id")
	} else if _, err := strconv.Atoi(c.ChannelID); err != nil {
		return errors.New("channel_id must be an integer encoded as a string")
	} else if c.OpaqueID == "" {
		return errors.New("missing opaque_user_id")
	} else if c.OpaqueID[0] == 'A' {
		return errors.New("anonymous user")
	} else if c.Role == "" {
		return errors.New("missing role")
	} else if c.Role != "broadcaster" && c.Role != "moderator" && c.Role != "viewer" {
		return errors.New("invalid role")
	} else {
		return nil
	}
}

func JWTMiddleware(secret string) (gin.HandlerFunc, error) {
	secretBuf, err := base64.URLEncoding.DecodeString(secret)
	if err != nil {
		return nil, fmt.Errorf("error decoding shared secret: %s", err)
	}

	algo := jwt.HS256(secretBuf)
	expHeader := jwt.NewHeader(algo)
	send401 := func(ctx *gin.Context, err error) {
		ctx.Header("WWW-Authenticate", `Bearer token_type="JWT"`)
		ctx.AbortWithStatusJSON(401, gin.H{"error": err.Error()})
	}
	return func(ctx *gin.Context) {
		if ctx.Request.Method == "OPTIONS" {
			ctx.Next()
			return
		}

		var h jwt.Header
		var claims jwtClaims
		if tok := ctx.GetHeader("Authorization"); !strings.HasPrefix(tok, "Bearer ") {
			send401(ctx, errors.New("missing Authorization header"))
		} else if err := jwt.DecodeAndValidate(&h, &claims, algo, []byte(tok)[7:]); err != nil {
			send401(ctx, err)
		} else if err = h.ValidateEqual(expHeader); err != nil {
			send401(ctx, err)
		} else if err = claims.validate(); err != nil {
			send401(ctx, err)
		} else {
			// `validate()` will have verified that `channel_id` can be parsed as int.
			channelID, _ := strconv.Atoi(claims.ChannelID)
			ctx.Set("jwt", true)
			ctx.Set("channel_id", channelID)
			ctx.Set("opaque_id", claims.OpaqueID)
			ctx.Set("role", claims.Role)
		}
		ctx.Next()
	}, nil
}
