package handlers

import (
	"time"
	"ting/model"

	"github.com/gin-gonic/gin"
)

func addQuestionHandlers(engine gin.IRouter, db *model.DB) {
	engine.OPTIONS("/questions", makeOptionsHandler("GET", "POST"))
	engine.GET("/questions", GetQuestionsHandler(db))
	engine.POST("/questions", PostQuestionHandler(db))

	engine.OPTIONS("/questions/:question_id", makeOptionsHandler("GET", "PATCH", "DELETE"))
	engine.GET("/questions/:question_id", GetQuestionHandler(db))
	engine.PATCH("/questions/:question_id", PatchQuestionHandler(db))
	engine.DELETE("/questions/:question_id", DeleteQuestionHandler(db))
}

// GET /questions
func GetQuestionsHandler(db *model.DB) gin.HandlerFunc {
	h := NewHandler(
		DefParam("channel_id", "int", "query", true, nil),
		DefParam("active_from", "time", "query", false, (*time.Time)(nil)),
		DefParam("active_until", "time", "query", false, (*time.Time)(nil)),
	)

	return h.Wrap(func(ctx *gin.Context) {
		if questions, err := db.ListQuestions(
			h.IntParam("channel_id"),
			h.TimeParam("active_from"),
			h.TimeParam("active_until"),
		); err != nil {
			replyError(ctx, err)
		} else {
			ctx.JSON(200, questions)
		}
	})
}

// POST /questions
func PostQuestionHandler(db *model.DB) gin.HandlerFunc {
	return func(ctx *gin.Context) {
		var newQuestion model.NewQuestion
		if jwtRole, found := ctx.Get("role"); found && jwtRole.(string) != "broadcaster" {
			ctx.JSON(403, gin.H{"error": "only broadcaster can make new questions"})
		} else if bindErr := ctx.BindJSON(&newQuestion); bindErr != nil {
			ctx.JSON(400, gin.H{"error": "JSON parse error"})
		} else if jwtChannelID, found := ctx.Get("channel_id"); found && jwtChannelID != newQuestion.ChannelID {
			ctx.JSON(403, gin.H{"error": "cannot create a question for another channel"})
		} else if question, err := db.CreateQuestion(newQuestion); err != nil {
			replyError(ctx, err)
		} else {
			ctx.JSON(200, question)
		}
	}
}

// GET /questions/:question_id
func GetQuestionHandler(db *model.DB) gin.HandlerFunc {
	h := NewHandler(
		DefParam("question_id", "int", "url", true, nil),
		DefParam("channel_id", "int", "query", true, nil),
		DefParam("user_id", "string", "query", false, nil),
	)

	return h.Wrap(func(ctx *gin.Context) {
		questionID := h.IntParam("question_id")
		channelID := h.IntParam("channel_id")

		opaqueID, found := h.StringOptParam("user_id")
		if _, jwtFound := ctx.Get("jwt"); jwtFound {
			var errMsg string
			if channelID != ctx.GetInt("channel_id") {
				errMsg = "channel_id mismatch"
			} else if !found {
				opaqueID = ctx.GetString("opaque_id")
			} else if opaqueID != ctx.GetString("opaque_id") {
				errMsg = "opaque_id mismatch"
			}
			if errMsg != "" {
				ctx.JSON(403, gin.H{"error": errMsg})
				return
			}
		}

		if question, err := db.FindQuestion(questionID); err != nil {
			replyError(ctx, err)
		} else if question == nil || question.ChannelID != channelID {
			ctx.Status(404)
		} else if opaqueID == "" {
			ctx.JSON(200, question)
		} else if err = question.ForUser(opaqueID); err != nil {
			replyError(ctx, err)
		} else {
			ctx.JSON(200, question)
		}
	})
}

// PATCH /questions/:question_id
func PatchQuestionHandler(db *model.DB) gin.HandlerFunc {
	h := NewHandler(
		DefParam("question_id", "int", "url", true, nil),
	)

	return h.Wrap(func(ctx *gin.Context) {
		var updates map[string]interface{}
		if jwtRole, found := ctx.Get("role"); found && jwtRole != "broadcaster" {
			ctx.JSON(403, gin.H{"error": "only broadcaster can modify questions"})
		} else if bindErr := ctx.BindJSON(&updates); bindErr != nil {
			ctx.JSON(400, gin.H{"error": "invalid JSON"})
		} else if q, err := db.FindQuestion(h.IntParam("question_id")); err != nil {
			replyError(ctx, err)
		} else if q == nil {
			ctx.JSON(404, gin.H{"error": "no such question"})
		} else if jwtChannelID, found := ctx.Get("channel_id"); found && jwtChannelID != q.ChannelID {
			ctx.JSON(403, gin.H{"error": "cannot modify a question from another channel"})
		} else if q, err := q.Update(updates); err != nil {
			replyError(ctx, err)
		} else {
			ctx.JSON(200, q)
		}
	})
}

// DELETE /questions/:question_id
func DeleteQuestionHandler(db *model.DB) gin.HandlerFunc {
	h := NewHandler(
		DefParam("question_id", "int", "url", true, nil),
	)

	return h.Wrap(func(ctx *gin.Context) {
		if jwtRole, found := ctx.Get("role"); found && jwtRole.(string) != "broadcaster" {
			ctx.JSON(403, gin.H{"error": "only broadcaster can delete questions"})
		} else if q, err := db.FindQuestion(h.IntParam("question_id")); err != nil {
			replyError(ctx, err)
		} else if q == nil {
			ctx.JSON(404, gin.H{"error": "no such question"})
		} else if jwtChannelID, found := ctx.Get("channel_id"); found && jwtChannelID != q.ChannelID {
			ctx.JSON(403, gin.H{"error": "cannot delete a question from another channel"})
		} else if err = q.Delete(); err != nil {
			replyError(ctx, err)
		} else {
			ctx.JSON(200, q)
		}
	})
}
