// +build test

package handlers

import (
	"fmt"
	"testing"
	"time"

	"ting/model"
	"ting/util"
	. "ting/util/types"
)

func (ht handlerTest) GetQuestionsTests(t *testing.T) {
	t.Run("None", func(t *testing.T) {
		channelID := ht.db.NewTestChannel(t, nil).ID
		url := fmt.Sprintf("/questions?channel_id=%d", channelID)
		if code, body := ht.GET(t, url); code != 200 {
			t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
		} else if body != `[]` {
			t.Fatalf("unexpected results (expected []): %s", body)
		}
	})

	t.Run("Some", func(t *testing.T) {
		channelID := ht.db.NewTestChannel(t, nil).ID
		qsOrig := []*model.Question{
			ht.db.NewTestQuestion(t, model.Attr{"ChannelID": channelID}),
			ht.db.NewTestQuestion(t, model.Attr{"ChannelID": channelID, "Active": true}),
		}

		// Create some potential false positives:
		badChannelID := ht.db.NewTestChannel(t, nil).ID
		ht.db.NewTestQuestion(t, model.Attr{"ChannelID": badChannelID})
		ht.db.NewTestQuestion(t, model.Attr{"ChannelID": badChannelID, "Active": true})

		qsExpected := []*model.Question{
			model.ParseQuestion(t, util.ToJSON(t, qsOrig[0])),
			model.ParseQuestion(t, util.ToJSON(t, qsOrig[1])),
		}
		// Paranoia:
		for i, qExp := range qsExpected {
			qOrig := qsOrig[i]
			if qExp.ID != qOrig.ID || qExp.ChannelID != qOrig.ChannelID {
				t.Fatalf("test question did not reparse correctly\nfound: %#v\nexpected: %#v", qExp, qOrig)
			}
		}

		url := fmt.Sprintf("/questions?channel_id=%d", channelID)
		code, body := ht.GET(t, url)
		if code != 200 {
			t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
		}
		qsActual := model.ParseQuestions(t, body)
		if len(qsActual) != len(qsExpected) {
			t.Fatalf("wrong number of results: %d; expected %d\nbody: %s", len(qsActual), len(qsExpected), body)
		}
		for i, qAct := range qsActual {
			qAct.AssertEqual(t, qsExpected[i])
		}
	})
}

func (ht handlerTest) GetQuestionTests(t *testing.T) {
	t.Run("404", func(t *testing.T) {
		code, body := ht.GET(t, "/questions/-1?channel_id=-1")
		if code != 404 {
			t.Errorf("unexpected HTTP status: %d; expected 404", code)
		}
		if len(body) != 0 {
			t.Errorf("unexpected reply body (expected empty)\nbody: %s", body)
		}
	})

	t.Run("WrongChannelID", func(t *testing.T) {
		q := ht.db.NewTestQuestion(t, nil)
		url := fmt.Sprintf("/questions/%d?channel_id=-1", q.ID)
		if code, body := ht.GET(t, url); code != 404 {
			t.Errorf("unexpected HTTP status: %d; expected 404", code)
		} else if len(body) != 0 {
			t.Errorf("unexpected reply body (expected empty):\n%s", body)
		}
	})

	t.Run("NoUser", func(t *testing.T) {
		q := ht.db.NewTestQuestion(t, nil)
		if q.Answered != nil {
			t.Fatalf("unexpected Answered field: %v\nUserAnswer: %#v", *q.Answered, q.UserAnswer)
		} else if q.UserAnswer != nil {
			t.Fatalf("unexpected UserAnswer (Answered is nil): %#v", *q.UserAnswer)
		}

		url := fmt.Sprintf("/questions/%d?channel_id=%d", q.ID, q.ChannelID)
		code, body := ht.GET(t, url)
		if code != 200 {
			t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
		}
		qActual := model.ParseQuestion(t, body)
		qActual.AssertEqual(t, q)
	})

	t.Run("UserNoAnswer", func(t *testing.T) {
		q := ht.db.NewTestQuestion(t, model.Attr{"UserAnswer": 0})
		if q.Answered == nil {
			t.Fatal("expected non-nil Answered field")
		} else if *q.Answered {
			t.Fatal("Answered field should not be true")
		} else if q.UserAnswer == nil {
			t.Fatalf("expected non-nil UserAnswer")
		} else if q.UserAnswer.AnswerID != nil {
			t.Fatalf("unexpected UserAnswer.AnswerID: %d; expected nil", *q.UserAnswer.AnswerID)
		}

		url := fmt.Sprintf("/questions/%d?channel_id=%d&user_id=%s", q.ID, q.ChannelID, q.UserAnswer.User.OpaqueID)
		code, body := ht.GET(t, url)
		if code != 200 {
			t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
		}
		qActual := model.ParseQuestion(t, body)
		qActual.AssertEqual(t, q)
	})

	t.Run("UserWithAnswer", func(t *testing.T) {
		q := ht.db.NewTestQuestion(t, model.Attr{"Active": true, "UserAnswer": 1})
		url := fmt.Sprintf("/questions/%d?channel_id=%d&user_id=%s", q.ID, q.ChannelID, q.UserAnswer.User.OpaqueID)
		code, body := ht.GET(t, url)
		if code != 200 {
			t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
		}
		qActual := model.ParseQuestion(t, body)
		qActual.AssertEqual(t, q)
	})

	t.Run("JWT", func(t *testing.T) {
		u := ht.db.NewTestUser(t, nil)
		q := ht.db.NewTestQuestion(t, model.Attr{"UserID": u.ID, "UserAnswer": 0})
		headers := makeJWTHeader(t, nil, StringMap{"opaque_user_id": u.OpaqueID, "channel_id": q.ChannelID})

		t.Run("OK", func(t *testing.T) {
			url := fmt.Sprintf("/jwt/questions/%d?channel_id=%d&user_id=%s", q.ID, q.ChannelID, u.OpaqueID)
			code, body, _ := request(t, ht.engine, "GET", url, "", headers)
			if code != 200 {
				t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
			}
			model.ParseQuestion(t, body).AssertEqual(t, q)
		})

		t.Run("InferUser", func(t *testing.T) {
			url := fmt.Sprintf("/jwt/questions/%d?channel_id=%d", q.ID, q.ChannelID)
			code, body, _ := request(t, ht.engine, "GET", url, "", headers)
			if code != 200 {
				t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
			} else if qReply := model.ParseQuestion(t, body); qReply.Answered == nil || *qReply.Answered != false {
				t.Fatalf(`wrong "answered" field: %v; expected false`, qReply.Answered)
			}
		})

		t.Run("UserMismatch", func(t *testing.T) {
			url := fmt.Sprintf("/jwt/questions/%d?channel_id=%d&user_id=%s", q.ID, q.ChannelID, u.OpaqueID+"_lol")
			code, body, _ := request(t, ht.engine, "GET", url, "", headers)
			if code != 403 {
				t.Fatalf("wrong HTTP status code: %d\nbody: %s", code, body)
			} else if reply := parseErrReply(t, body); reply.Error != "opaque_id mismatch" {
				t.Fatalf("wrong error message: %q", reply.Error)
			}
		})

		t.Run("ChannelMismatch", func(t *testing.T) {
			url := fmt.Sprintf("/jwt/questions/%d?channel_id=%d&user_id=%s", q.ID, q.ChannelID+1, u.OpaqueID)
			code, body, _ := request(t, ht.engine, "GET", url, "", headers)
			if code != 403 {
				t.Fatalf("wrong HTTP status code: %d\nbody: %s", code, body)
			} else if reply := parseErrReply(t, body); reply.Error != "channel_id mismatch" {
				t.Fatalf("wrong error message: %q", reply.Error)
			}
		})
	})
}

func (ht handlerTest) PostQuestionTests(t *testing.T) {
	t.Run("Blank", func(t *testing.T) {
		if code, body := ht.POST(t, "/questions", ""); code != 400 {
			t.Fatalf("wrong HTTP response code: %d; expected 400\nbody: %s", code, body)
		} else if reply := parseErrReply(t, body); reply.Error != "JSON parse error" {
			t.Fatalf("unexpected error message: %q", reply.Error)
		}
	})

	t.Run("Inactive", func(t *testing.T) {
		channel := ht.db.NewTestChannel(t, nil)
		reqBody := fmt.Sprintf(`{
				"channel_id": %d,
				"text": "don't answer me",
				"active_from": null,
				"answers": [
					{"text": "answer one"},
					{"text": "answer two", "is_correct": true},
					{"text": "answer three", "is_correct": false}
				]
			}`, channel.ID)

		code, body := ht.POST(t, "/questions", reqBody)
		if code != 200 {
			t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
		}
		q := model.ParseQuestion(t, body)
		q.SimpleCheck(t, "don't answer me", nil, nil,
			[]string{"answer one", "answer two", "answer three"}, []int{1})
		for i, answer := range q.Answers {
			correct := i == 1
			if answer.IsCorrect != correct {
				t.Fatalf("wrong is_correct for answer %d/%d: %v; expected %v",
					i, len(q.Answers), answer.IsCorrect, correct)
			}
		}
	})

	t.Run("Active", func(t *testing.T) {
		channel := ht.db.NewTestChannel(t, nil)
		now := time.Now().UTC()
		from := now.Add(-1 * time.Hour)
		until := now.Add(+1 * time.Hour)
		reqBody := fmt.Sprintf(`{
				"channel_id": %d,
				"text": "don't answer me",
				"active_from": %q,
				"active_until": %q,
				"answers": [
					{"text": "answer one"},
					{"text": "answer two", "is_correct": true},
					{"text": "answer three", "is_correct": false}
				]
			}`, channel.ID, from.Format(time.RFC3339), until.Format(time.RFC3339))

		code, body := ht.POST(t, "/questions", reqBody)
		if code != 200 {
			t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
		}
		q := model.ParseQuestion(t, body)
		q.SimpleCheck(t, "don't answer me", &from, &until,
			[]string{"answer one", "answer two", "answer three"}, []int{1})
	})

	t.Run("Duplicate", func(t *testing.T) {
		q := ht.db.NewTestQuestion(t, model.Attr{"Active": true})
		reqBody := fmt.Sprintf(
			`{"channel_id": %d, "text": %q, "active_from": %q, "active_until": %q, `+
				`"answers": [{"text": "answer1"}, {"text": "answer2"}]}`,
			q.ChannelID, q.Text, q.ActiveFrom.Format(time.RFC3339), q.ActiveUntil.Format(time.RFC3339),
		)
		if code, body := ht.POST(t, "/questions", reqBody); code != 400 {
			t.Fatalf("unexpected HTTP status code: %d\nbody: %s", code, body)
		} else if reply := parseErrReply(t, body); reply.Error != "question overlaps existing questions" {
			t.Fatalf("wrong error message: %q", reply.Error)
		} else if len(reply.Conflicts) != 1 || reply.Conflicts[0] != q.ID {
			t.Fatalf("wrong conflicting question IDs: %v; expected [%d]", reply.Conflicts, q.ID)
		}
	})

	t.Run("JWT", func(t *testing.T) {
		channel := ht.db.NewTestChannel(t, nil)
		reqBody := fmt.Sprintf(`{"channel_id":%d,"text":"lol","answers":[{"text":"1"}]}`, channel.ID)
		jwtAttrs := StringMap{"role": "broadcaster", "channel_id": channel.ID}

		t.Run("WrongRole", func(t *testing.T) {
			headers := makeJWTHeader(t, nil, jwtAttrs.With("role", "viewer"))
			code, body, _ := request(t, ht.engine, "POST", "/jwt/questions", reqBody, headers)
			if code != 403 {
				t.Fatalf("wrong HTTP status code: %d\nbody: %s", code, body)
			} else if reply := parseErrReply(t, body); reply.Error != "only broadcaster can make new questions" {
				t.Fatalf("wrong error message: %q", reply.Error)
			}
		})

		t.Run("WrongChannel", func(t *testing.T) {
			headers := makeJWTHeader(t, nil, jwtAttrs.With("channel_id", -1))
			code, body, _ := request(t, ht.engine, "POST", "/jwt/questions", reqBody, headers)
			if code != 403 {
				t.Fatalf("wrong HTTP status code: %d\nbody: %s", code, body)
			} else if reply := parseErrReply(t, body); reply.Error != "cannot create a question for another channel" {
				t.Fatalf("wrong error message: %q", reply.Error)
			}
		})

		t.Run("OK", func(t *testing.T) {
			headers := makeJWTHeader(t, nil, jwtAttrs)
			if code, body, _ := request(t, ht.engine, "POST", "/jwt/questions", reqBody, headers); code != 200 {
				t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
			}
		})
	})
}

func (ht handlerTest) PatchQuestionTests(t *testing.T) {
	qPre := ht.db.NewTestQuestion(t, nil)
	url := fmt.Sprintf("/questions/%d", qPre.ID)
	updates := `{"text": "new question text"}`

	t.Run("200", func(t *testing.T) {
		if code, body := ht.PATCH(t, url, updates); code != 200 {
			t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
		} else if qPost := model.ParseQuestion(t, body); qPost.Text != "new question text" {
			t.Fatalf("wrong question text in response: %q", qPost.Text)
		}
	})

	t.Run("JWT", func(t *testing.T) {
		jwtAttrs := StringMap{"role": "broadcaster", "channel_id": qPre.ChannelID}

		t.Run("WrongRole", func(t *testing.T) {
			headers := makeJWTHeader(t, nil, jwtAttrs.With("role", "moderator"))
			code, body, _ := request(t, ht.engine, "PATCH", "/jwt"+url, updates, headers)
			if code != 403 {
				t.Fatalf("wrong HTTP status code: %d\nbody: %s", code, body)
			} else if reply := parseErrReply(t, body); reply.Error != "only broadcaster can modify questions" {
				t.Fatalf("wrong error message: %q", reply.Error)
			}
		})

		t.Run("WrongChannel", func(t *testing.T) {
			headers := makeJWTHeader(t, nil, jwtAttrs.With("channel_id", -1))
			code, body, _ := request(t, ht.engine, "PATCH", "/jwt"+url, updates, headers)
			if code != 403 {
				t.Fatalf("wrong HTTP status code: %d\nbody: %s", code, body)
			} else if reply := parseErrReply(t, body); reply.Error != "cannot modify a question from another channel" {
				t.Fatalf("wrong error message: %q", reply.Error)
			}
		})

		t.Run("OK", func(t *testing.T) {
			headers := makeJWTHeader(t, nil, jwtAttrs)
			if code, body, _ := request(t, ht.engine, "PATCH", "/jwt"+url, updates, headers); code != 200 {
				t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
			}
		})
	})
}

func (ht handlerTest) DeleteQuestionTests(t *testing.T) {
	t.Run("404", func(t *testing.T) {
		code, body := ht.DELETE(t, "/questions/-1")
		if code != 404 {
			t.Fatalf("unexpected HTTP status: %d\nbody: %s", body)
		} else if reply := parseErrReply(t, body); reply.Error != "no such question" {
			t.Fatalf("wrong error message: %q", body)
		}
	})

	t.Run("200", func(t *testing.T) {
		q := ht.db.NewTestQuestion(t, nil)
		url := fmt.Sprintf("/questions/%d", q.ID)
		code, body := ht.DELETE(t, url)
		if code != 200 {
			t.Fatalf("unexpected failure: HTTP %d\nbody: %s", code, body)
		}
		model.ParseQuestion(t, body).AssertEqual(t, q)
	})

	t.Run("JWT", func(t *testing.T) {
		t.Run("WrongRole", func(t *testing.T) {
			q := ht.db.NewTestQuestion(t, nil)
			headers := makeJWTHeader(t, nil, StringMap{"role": "viewer", "channel_id": q.ChannelID})
			url := fmt.Sprintf("/jwt/questions/%d", q.ID)
			code, body, _ := request(t, ht.engine, "DELETE", url, "", headers)
			if code != 403 {
				t.Fatalf("wrong HTTP status code: %d\nbody: %s", code, body)
			} else if reply := parseErrReply(t, body); reply.Error != "only broadcaster can delete questions" {
				t.Fatalf("wrong error message: %q", reply.Error)
			}
		})

		t.Run("WrongChannel", func(t *testing.T) {
			q := ht.db.NewTestQuestion(t, nil)
			headers := makeJWTHeader(t, nil, StringMap{"role": "broadcaster", "channel_id": -1})
			url := fmt.Sprintf("/jwt/questions/%d", q.ID)
			code, body, _ := request(t, ht.engine, "DELETE", url, "", headers)
			if code != 403 {
				t.Fatalf("wrong HTTP status code: %d\nbody: %s", code, body)
			} else if reply := parseErrReply(t, body); reply.Error != "cannot delete a question from another channel" {
				t.Fatalf("wrong error message: %q", reply.Error)
			}
		})

		t.Run("OK", func(t *testing.T) {
			q := ht.db.NewTestQuestion(t, nil)
			headers := makeJWTHeader(t, nil, StringMap{"role": "broadcaster", "channel_id": q.ChannelID})
			url := fmt.Sprintf("/jwt/questions/%d", q.ID)
			if code, body, _ := request(t, ht.engine, "DELETE", url, "", headers); code != 200 {
				t.Fatalf("unexpected error: HTTP %d\nbody: %s", code, body)
			}
		})
	})
}
