// +build test

package handlers

import (
	"encoding/json"
	"net/http"
	"net/http/httptest"
	"strings"
	"testing"

	"ting/util"

	"github.com/gin-gonic/gin"
)

func request(t *testing.T, engine *gin.Engine, method, url, reqBody string, reqHeaders map[string]string) (code int, body string, headers http.Header) {
	t.Helper()
	response := httptest.NewRecorder()
	request, err := http.NewRequest(method, url, strings.NewReader(reqBody))
	if err != nil {
		t.Fatalf("unexpected error creating request (%s %q): %s", method, url, err)
	}
	for header, value := range reqHeaders {
		request.Header.Set(header, value)
	}
	engine.ServeHTTP(response, request)
	return response.Code, response.Body.String(), response.HeaderMap
}

func dumpResponse(t *testing.T, code int, body string, headers http.Header) {
	t.Helper()
	t.Logf("code: %d", code)
	if len(headers) == 0 {
		t.Logf("headers: (none)")
	} else {
		t.Logf("headers (%d):", len(headers))
		for header, value := range headers {
			t.Logf("    %q (%d): %q", header, len(value), strings.Join(value, ","))
		}
	}
	var j map[string]interface{}
	if err := json.Unmarshal([]byte(body), &j); err != nil {
		t.Logf("body (non-JSON):")
		t.Log(body)
	} else {
		t.Logf("body (JSON):")
		t.Log(util.ToJSONPP(t, j))
	}
}

func dumpResponseIfFailed(t *testing.T, code int, body string, headers http.Header) {
	if !t.Failed() {
		return
	}
	t.Helper()
	dumpResponse(t, code, body, headers)
}
