package logging

import (
	"fmt"
	"time"

	"ting/util/set"

	"github.com/sirupsen/logrus"
	"golang.org/x/sys/unix"
)

func isTerm() bool {
	_, err := unix.IoctlGetTermios(1, unix.TCGETS)
	return err == nil
}

func fmtTime(t time.Time) string {
	return t.UTC().Format(time.RFC3339)
}

var Formats set.StringSet
var Levels set.StringSet
var formatters map[string]logrus.Formatter

func init() {
	Formats = set.NewStringSet("text", "plain", "json")
	Levels = set.NewStringSet("panic", "fatal", "error", "warn", "info", "debug")
	formatters = map[string]logrus.Formatter{
		"text":  &textFormatter{color: true},
		"plain": &textFormatter{color: false},
		"auto":  &textFormatter{color: isTerm()},
		"json":  &jsonFormatter{},
	}
}

func Init(format, level string) error {
	if logrusLevel, err := logrus.ParseLevel(level); err != nil {
		return fmt.Errorf("invalid logging level: %q", level)
	} else {
		logrus.SetLevel(logrusLevel)
	}

	if formatter, found := formatters[format]; !found {
		return fmt.Errorf("invalid logging format: %q", format)
	} else {
		logrus.SetFormatter(formatter)
	}

	return nil
}
