package main

import (
	"fmt"
	"net/http/httputil"
	"time"

	"github.com/gin-gonic/gin"
	"github.com/sirupsen/logrus"
)

func RecoverMW(ctx *gin.Context) {
	defer func() {
		e := recover()
		if e == nil {
			return
		}
		reqDump, _ := httputil.DumpRequest(ctx.Request, false)
		logrus.WithFields(map[string]interface{}{
			"panic":   e,
			"request": string(reqDump),
			"msg":     "caught panic",
		}).Infof("caught panic: %#v; request: %q", e, string(reqDump))
		ctx.AbortWithStatus(500)
	}()
	ctx.Next()
}

func LogRequestMW(ctx *gin.Context) {
	start := time.Now()
	desc := fmt.Sprintf("%s %s <- %s", ctx.Request.Method, ctx.Request.RequestURI, ctx.ClientIP())
	log := logrus.WithFields(map[string]interface{}{
		"client": ctx.ClientIP(),
		"method": ctx.Request.Method,
		"uri":    ctx.Request.RequestURI,
		"msg":    "request",
	})
	log.Info(desc)

	ctx.Next()

	duration := time.Since(start)
	status := ctx.Writer.Status()
	log.WithFields(map[string]interface{}{
		"status":   status,
		"duration": duration.Seconds(),
		"msg":      "response",
	}).Infof("HTTP %d in %s -> %s", status, duration, desc)
}
