package model

import (
	"database/sql"
	"encoding/json"

	. "ting/util/types"
)

type Channel struct {
	ID       int             `db:"id"        json:"id"`
	Settings ChannelSettings `db:"settings"  json:"settings"`

	CurrentQuestionID *int      `db:"-"  json:"current_question_id"`
	CurrentQuestion   *Question `db:"-"  json:"current_question"`

	db *DB
}

type ChannelSettings struct {
	Theme         string    `json:"theme"`
	ThemeSettings StringMap `json:"theme_settings"`
}

// Implements `database/sql.Scan`.
func (s *ChannelSettings) Scan(src interface{}) error {
	return json.Unmarshal(src.([]byte), s)
}

func (db *DB) FindOrCreateChannel(id int) (*Channel, Error) {
	var channel Channel

	err := db.Get(&channel, `INSERT INTO "channels" ("id") VALUES ($1) ON CONFLICT DO NOTHING RETURNING *`, id)
	if err == sql.ErrNoRows {
		// When upsert finds existing row, "DO NOTHING" makes it return 0 rows despite "RETURNING *".
		err = db.Get(&channel, `SELECT * FROM "channels" WHERE "id" = $1 LIMIT 1`, id)
	}
	if err != nil {
		return nil, DBError(err)
	}
	channel.db = db

	return &channel, nil
}

func (c *Channel) WithCurrentQuestion() Error {
	if c.CurrentQuestion != nil {
		return nil
	}

	var qID int
	if err := c.db.Get(&qID,
		`SELECT "id" FROM "questions" WHERE "channel_id" = $1 `+
			`AND "active_from"  <= (NOW() AT TIME ZONE 'UTC') `+
			`AND "active_until" >= (NOW() AT TIME ZONE 'UTC') LIMIT 1`,
		c.ID,
	); err == sql.ErrNoRows {
		return nil
	} else if err != nil {
		return DBErrorf("error determining current question ID: %s", err)
	} else if q, qErr := c.db.FindQuestion(qID); qErr != nil {
		return qErr.Prefix("error fetching current question (%d)", qID)
	} else {
		c.CurrentQuestionID = &qID
		c.CurrentQuestion = q
		return nil
	}
}

func (c *Channel) ForUser(opaqueID string) Error {
	if c.CurrentQuestion == nil {
		return nil
	}
	return c.CurrentQuestion.ForUser(opaqueID)
}

func (c *Channel) Update(updates map[string]interface{}) (*Channel, Error) {
	// TODO: Invalid/mismatched field checking? Is it even worth it? x_x

	var ok bool
	if settingsV, found := updates["settings"]; !found {
		return c, nil
	} else if updates, ok = settingsV.(map[string]interface{}); !ok {
		return nil, UserErrorf("settings: must be an object")
	}

	if themeV, found := updates["theme"]; found {
		if theme, ok := themeV.(string); !ok {
			return nil, UserErrorf("settings.theme: must be a string")
		} else if theme != c.Settings.Theme {
			c.Settings.Theme = theme
			c.Settings.ThemeSettings = StringMap{}
		}
	}

	if themeSettingsV, found := updates["theme_settings"]; found {
		if themeSettings, ok := themeSettingsV.(map[string]interface{}); !ok {
			return nil, UserErrorf("settings.theme_settings: must be an object")
		} else {
			c.Settings.ThemeSettings = StringMap(themeSettings)
		}
	}

	var settingsJSON string
	if buf, err := json.Marshal(&c.Settings); err != nil {
		return nil, DBErrorf("error serializing channel theme settings to JSON: %s", err)
	} else {
		settingsJSON = string(buf)
	}

	if _, err := c.db.Exec(
		`UPDATE "channels" SET "settings" = $1::jsonb WHERE "id" = $2`,
		settingsJSON, c.ID,
	); err != nil {
		return nil, DBErrorf("error updating channel %d: %s", c.ID, err)
	}

	result, err := c.db.FindOrCreateChannel(c.ID)
	if err != nil {
		return nil, err
	}
	return result, result.WithCurrentQuestion()
}
