// +build test

package model

import (
	"testing"

	"ting/util"
	. "ting/util/types"
)

func TestChannel(t *testing.T) {
	db := InitTestDB(t)
	defer db.Close()

	t.Run("FindOrCreate", findOrCreateChannelTests(db))
	t.Run("WithCurrentQuestion", channelWithCurrentQuestionTests(db))
	t.Run("Update", updateChannelTests(db))
}

func getChannel(t *testing.T, db *DB, id int) (c *Channel) {
	var err error
	if c, err = db.FindOrCreateChannel(id); err != nil {
		t.Fatalf("unexpected error: %s", err)
	} else if c == nil {
		t.Fatal("nil channel despite nil error")
	} else if c.ID != id {
		t.Fatalf("channel has wrong ID: %d; expected %d", c.ID, id)
	}
	return c
}

func findOrCreateChannelTests(db *DB) func(t *testing.T) {
	return func(t *testing.T) {
		t.Run("New", func(t *testing.T) {
			id := db.NewTestChannel(t, nil).ID
			if _, err := db.Exec(`DELETE FROM "channels" WHERE "id" = $1`, id); err != nil {
				t.Fatalf("error deleting new channel %d: %s", id, err)
			}
			getChannel(t, db, id).AssertEqual(t, &Channel{
				ID: id,
				db: db,
			})
		})

		t.Run("NoSettings", func(t *testing.T) {
			c := db.NewTestChannel(t, nil)
			getChannel(t, db, c.ID).AssertEqual(t, c)
		})

		t.Run("WithSettings", func(t *testing.T) {
			settings := ChannelSettings{
				Theme:         "my-channel-theme",
				ThemeSettings: StringMap{"foo": "FOO", "bar": "BAR", "baz": "BAZ"},
			}
			c := db.NewTestChannel(t, Attr{"Settings": settings})
			if diffs := util.StructDiff(c.Settings, settings); len(diffs) > 0 {
				t.Fatalf("test channel %d created with wrong settings: %s", c.ID, util.DiffErrorMessage(diffs, 1))
			}
			getChannel(t, db, c.ID).AssertEqual(t, c)
		})
	}
}

func channelWithCurrentQuestionTests(db *DB) func(t *testing.T) {
	return func(t *testing.T) {
		t.Run("None", func(t *testing.T) {
			c := db.NewTestChannel(t, nil)
			c.AssertCurrentQuestion(t, nil)

			if err := c.WithCurrentQuestion(); err != nil {
				t.Fatalf("unexpected error getting current question: %q", err.Error())
			}
			c.AssertCurrentQuestion(t, nil)
		})

		t.Run("One", func(t *testing.T) {
			c := db.NewTestChannel(t, nil)
			c.AssertCurrentQuestion(t, nil)

			q := db.NewTestQuestion(t, Attr{"ChannelID": c.ID, "Active": true})
			if err := c.WithCurrentQuestion(); err != nil {
				t.Fatalf("unexpected error getting current question: %q", err.Error())
			}
			c.AssertCurrentQuestion(t, q)
		})
	}
}

func updateChannelTests(db *DB) func(t *testing.T) {
	return func(t *testing.T) {
		c := db.NewTestChannel(t, nil)
		cExp := &Channel{
			ID: c.ID,
			Settings: ChannelSettings{
				Theme:         "new theme",
				ThemeSettings: StringMap{"foo": "bar"},
			},
			db: c.db,
		}

		cResult, err := c.Update(map[string]interface{}{
			"settings": map[string]interface{}{
				"theme":          "new theme",
				"theme_settings": map[string]interface{}{"foo": "bar"},
			},
		})
		if err != nil {
			t.Fatalf("unexpected error updating channel %d: %s", c.ID, err)
		}
		cResult.AssertEqual(t, cExp)
	}
}
