package model

import (
	"encoding/json"
	"fmt"
)

type ErrorKind int

const (
	DBErrorKind ErrorKind = iota
	UserErrorKind
)

type Error interface {
	Kind() ErrorKind
	Meta() map[string]interface{}
	WithMeta(string, interface{}) Error
	Prefix(string, ...interface{}) Error
	error
	json.Marshaler
}

type modelError struct {
	kind ErrorKind
	msg  string
	meta map[string]interface{}
}

func newError(kind ErrorKind, msg string) Error {
	return modelError{
		kind: kind,
		msg:  msg,
		meta: map[string]interface{}{"error": msg},
	}
}

func DBError(err error) Error {
	return newError(DBErrorKind, err.Error())
}

func DBErrorf(fmtStr string, fmtArgs ...interface{}) Error {
	return newError(DBErrorKind, fmt.Sprintf(fmtStr, fmtArgs...))
}

func UserError(err error) Error {
	return newError(UserErrorKind, err.Error())
}

func UserErrorf(fmtStr string, fmtArgs ...interface{}) Error {
	return newError(UserErrorKind, fmt.Sprintf(fmtStr, fmtArgs...))
}

func (err modelError) WithMeta(k string, v interface{}) Error {
	err.meta[k] = v
	return err
}

func (err modelError) Prefix(fmtStr string, fmtArgs ...interface{}) Error {
	err.msg = fmt.Sprintf(fmtStr, fmtArgs...) + ": " + err.msg
	return err
}

func (err modelError) Kind() ErrorKind {
	return err.kind
}

func (err modelError) Meta() map[string]interface{} {
	return err.meta
}

// Implements builtin `error` interface.
func (err modelError) Error() string {
	return err.msg
}

// Implements `encoding/json.Marshaler`.
func (err modelError) MarshalJSON() ([]byte, error) {
	return json.Marshal(err.meta)
}
