// +build test

// Brace yourself for the most braindead and tedious mock code you've ever seen. I am tired and not sorry.
// Thankfully, the object model is yet small.

package model

import (
	"encoding/json"
	"fmt"
	"math/rand"
	"testing"
	"time"

	"github.com/malisit/kolpa"
	uuid "github.com/satori/go.uuid"
)

const TestPostgresURL = "postgres://ting@localhost:5432/ting_test?sslmode=disable"

var g kolpa.Generator

func init() {
	rand.Seed(time.Now().UnixNano())
	g = kolpa.C()
}

func InitTestDB(t *testing.T) *DB {
	t.Helper()
	db, err := InitDB(TestPostgresURL)
	if err != nil {
		t.Fatalf("error connecting to test db: %s", err)
	}
	return db
}

// Attributes (all optional):
//     ID: int
//     Settings: a `ChannelSettings` struct
func (db *DB) NewTestChannel(t *testing.T, attr Attr) *Channel {
	t.Helper()
	if attr == nil {
		attr = make(Attr, 0)
	}

	id := attr.Int("ID", -1)

	settings := attr.Get("Settings", ChannelSettings{}).(ChannelSettings)
	var settingsJSON string
	if buf, err := json.Marshal(&settings); err != nil {
		t.Fatalf("error serializing channel settings to JSON: %s; settings: %#v", err, settings)
	} else {
		settingsJSON = string(buf)
	}

	var channel Channel
	var err error
	if id != -1 {
		err = db.Get(&channel,
			`INSERT INTO "channels" ("id", "settings") VALUES ($1, $2::jsonb) RETURNING *`,
			id, settingsJSON,
		)
	} else {
		// Tests (and tests only) generate guaranteed-unique IDs using `SELECT MAX(id)+1 FROM "channels"`,
		// which is racy because the inferred lock level does not lock out other _readers_,
		// i.e. multiple readers can get the same `MAX(id)+1` value concurrently, and it's their inserts
		//      that are executed atomically but with the same conflicting ID.
		// The only way to lock out other _readers_ is with an `ACCESS EXCLUSIVE` lock, which itself must be
		// obtained within a transaction. (Again, this is only necessary in tests, so fuck performance.)
		tx, txErr := db.Beginx()
		if txErr != nil {
			t.Fatalf("error initiating transaction: %s", txErr)
		}
		defer tx.Rollback()
		if _, err = tx.Exec(`LOCK TABLE ONLY "channels" IN ACCESS EXCLUSIVE MODE`); err != nil {
			t.Fatalf(`error locking "channels" table: %s`, err)
		} else if err = tx.Get(&channel,
			`INSERT INTO "channels" ("id", "settings") SELECT COALESCE(MAX("id"),0)+1, $1::jsonb FROM "channels" RETURNING *`,
			settingsJSON,
		); err != nil {
			t.Fatalf("error creating channel with inferred id: %s", err)
		} else {
			err = tx.Commit()
		}
	}
	if err != nil {
		t.Fatalf("error creating channel: %s", err)
	}
	channel.db = db
	return &channel
}

// Attributes (all optional):
//     ChannelID: int
//     Text: string
//     Active: bool; if true, ActiveFrom and ActiveUntil are set to now -/+ 1 hour
//     ActiveFrom, ActiveUntil: time.Time, *time.Time, or RFC3339 timestamp as string
//     Settings: a QuestionSettings struct
//     Answers:
//         int: number of Answers to generate
//         []Attr: attributes to use for each answer; length determines number of answers;
//                 Answer attrs: Text (string), IsCorrect (bool), EmoteID (int), Match (string)
//     UserID: int; set Answered and UserAnswer for this user in result;
//             if "UserAnswer" attribute is not given, Question.Answered will be false
//     UserAnswer: int; set user answer to nth answer; 0 -> not answered
//                 if "UserID" attribute is not given, a new User will be made
func (db *DB) NewTestQuestion(t *testing.T, attr Attr) *Question {
	t.Helper()
	if attr == nil {
		// Allow `nil` for arg, allocate to avoid nil reference panics.
		attr = make(Attr, 0)
	}

	var newQ NewQuestion

	newQ.ChannelID = attr.Int("ChannelID", func() int {
		return db.NewTestChannel(t, nil).ID
	})
	newQ.Text = attr.String("Text", g.LoremSentence)

	if attr.Bool("Active", false) {
		now := time.Now().UTC()
		from := now.Add(-1 * time.Hour)
		until := now.Add(+1 * time.Hour)
		newQ.ActiveFrom = &from
		newQ.ActiveUntil = &until
	}
	if from := attr.Timep("ActiveFrom"); from != nil {
		newQ.ActiveFrom = from
	}
	if until := attr.Timep("ActiveUntil"); until != nil {
		newQ.ActiveUntil = until
	}

	newQ.Settings = attr.Get("Settings", QuestionSettings{}).(QuestionSettings)

	var answersAttrs []Attr
	genAnswers := 2 + rand.Intn(4) // 2 to 5
	if ans, found := attr["Answers"]; found {
		switch v := ans.(type) {
		case int:
			genAnswers = v
		case []Attr:
			answersAttrs = v
			genAnswers = 0
		default:
			panic(fmt.Sprintf(`"Answers" attribute must be either []Attr or count (int); found %T: %#v`, ans, ans))
		}
	}
	for i := 0; i < genAnswers; i++ {
		answersAttrs = append(answersAttrs, nil)
	}

	newQ.Answers = make([]NewAnswer, len(answersAttrs))
	for i, ansAttrs := range answersAttrs {
		if ansAttrs == nil {
			ansAttrs = make(Attr, 0)
		}
		newQ.Answers[i] = NewAnswer{
			Text:      ansAttrs.String("Text", g.LoremSentence),
			EmoteID:   ansAttrs.Intp("EmoteID", nil),
			Match:     ansAttrs.Stringp("Match", nil),
			IsCorrect: ansAttrs.Bool("IsCorrect", i == 0),
		}
	}

	q, err := db.CreateQuestion(newQ)
	if err != nil {
		t.Fatalf("failed to create test question: %s", err.Error())
	}

	userID := attr.Intp("UserID", nil)
	userAnswer := attr.Intp("UserAnswer", nil)
	if userID == nil && userAnswer == nil {
		return q
	}

	var user *User
	if userID == nil {
		user = db.NewTestUser(t, nil)
	} else {
		var err Error
		if user, err = db.FindUser(*userID); err != nil {
			t.Fatalf("error looking up user %d: %s", *userID, err.Error())
		}
	}

	if userAnswer != nil && *userAnswer > 0 {
		uaID := q.AnswerIDs[*userAnswer-1]
		if _, err := q.AddUserAnswer(user.OpaqueID, uaID, true); err != nil {
			t.Fatalf("error adding answer %d for user %d: %s", uaID, user.ID, err.Error())
		}
	}

	// Modified question; just re-fetch it.
	if q, err = db.FindQuestion(q.ID); err != nil {
		t.Fatalf("error re-fetching question %d: %s", q.ID, err.Error())
	} else if err = q.ForUser(user.OpaqueID); err != nil {
		t.Fatalf("error re-fetching question %d with user %d's answer: %s", q.ID, user.ID, err.Error())
	}

	return q
}

// Attributes (all optional):
//     OpaqueID: string (default: UUID)
//     TwitchID: string (default: "twitch:"+opaque)
//     Useranme: string (default: "user:"+opaque)
func (db *DB) NewTestUser(t *testing.T, attr Attr) *User {
	t.Helper()
	if attr == nil {
		attr = make(Attr, 0)
	}

	opaqueID := attr.String("OpaqueID", func() string { return "U-" + uuid.NewV4().String() })
	twitchID := attr.String("TwitchID", "twitch:"+opaqueID)
	username := attr.String("Username", "user:"+opaqueID)

	var user User
	if err := db.Get(&user,
		`INSERT INTO "users" ("opaque_id", "twitch_id", "username") VALUES ($1, $2, $3) RETURNING *`,
		opaqueID, twitchID, username,
	); err != nil {
		t.Fatalf("error creating user %q: %s", opaqueID, err)
	}
	user.db = db
	return &user
}
