// +build test

package model

import (
	"reflect"
	"time"

	. "ting/util/types"
)

type Attr StringMap

// The declaration `type Attr StringMap` does not inherit any methods from `StringMap`.
// (Struct embedding would, but then we'd need pointer receivers and a field used everywhere, which is even messier.)

// Copy() returns a (shallow) copy.
func (m Attr) Copy() Attr {
	return Attr(StringMap(m).Copy())
}

// With(key, value) returns a (shallow) copy with the new key/value.
func (m Attr) With(k string, v interface{}) Attr {
	return Attr(StringMap(m).With(k, v))
}

// Get(key, default) looks up the value of an attribute, backing off to a default value if it's not found.
// `default` can be the default value itself, a function of no arguments that returns it, or nil.
// Note that if `default` is `nil` and the attribute is not found, the expected type cannot be inferred,
// so the returned value will be the nil empty interface.
func (m Attr) Get(k string, f interface{}) interface{} {
	if m != nil {
		if x, found := m[k]; found {
			return x
		}
	}
	if f == nil {
		return nil
	}

	fV := reflect.ValueOf(f)
	if reflect.TypeOf(f).Kind() == reflect.Func {
		fV = fV.Call(nil)[0]
	}
	return fV.Interface()
}

// Get(key, default) is like Get(), but makes sure to return a pointer.
// If the attribute is not found and the `default` argument is or returns a non-pointer, a pointer to it is returned.
// As with `Get()`, the expected type cannot be inferred if `default` is `nil`, so the returned `nil` has type `interface{}`.
func (m Attr) Getp(k string, f interface{}) interface{} {
	if v := m.Get(k, f); v == nil {
		return nil
	} else if vT := reflect.TypeOf(v); vT.Kind() == reflect.Ptr {
		return v
	} else {
		p := reflect.New(vT)
		reflect.Indirect(p).Set(reflect.ValueOf(v))
		return p.Interface()
	}
}

// Helpers for various types:

func (m Attr) String(k string, def interface{}) string {
	return m.Get(k, def).(string)
}

func (m Attr) Stringp(k string, def interface{}) *string {
	if v := m.Getp(k, def); v != nil {
		return v.(*string)
	}
	return nil
}

func (m Attr) Int(k string, def interface{}) int {
	return m.Get(k, def).(int)
}

func (m Attr) Intp(k string, def interface{}) *int {
	if v := m.Getp(k, def); v != nil {
		return v.(*int)
	}
	return nil
}

func (m Attr) Bool(k string, def interface{}) bool {
	return m.Get(k, def).(bool)
}

func (m Attr) Timep(k string) *time.Time {
	v := m.Get(k, nil)
	if v == nil {
		return nil
	}
	switch t := v.(type) {
	case *time.Time:
		return t
	case time.Time:
		return &t
	case string:
		if tt, err := time.Parse(time.RFC3339, t); err == nil {
			return &tt
		}
	}
	return nil
}
